<?php

namespace V360\ExpoPusher;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Route;
use ExponentPhpSDK\Expo;
use ExponentPhpSDK\ExpoRegistrar;
use ExponentPhpSDK\ExpoRepository;
use V360\ExpoPusher\ExpoChannel;
use V360\ExpoPusher\Repositories\ExpoPusherDatabaseDriver;
use V360\ExpoPusher\Repositories\ExpoPusherFileDriver;
use V360\ExpoPusher\Repositories\ExpoPusherRedisDriver;

class ExpoPusherServiceProvider extends ServiceProvider
{
  public function register()
  {
    //
    //merge config
    $this->mergeConfigFrom(__DIR__ . '/../config/config.php', 'expo_pusher');

    //register facade
    $this->app->bind(ExpoRepository::class, get_class($this->getInterestsDriver()));
  }

  public function boot()
  {
    //
    //config
    if ($this->app->runningInConsole()) {

      // publish the configs
      $this->publishes([
        __DIR__ . '/../config/config.php' => config_path('expo_pusher.php'),
      ], 'config');

      // publish the migration
      if (!class_exists('CreateV360ExpoPusherSubscribersTable')) {
        $this->publishes([
          __DIR__ . '/../database/migrations/create_v360_expo_pusher_subscribers_table.php.stub' => database_path('migrations/' . date('Y_m_d_His', time()) . '_create_v360_expo_pusher_subscribers_table.php'),
          // you can add any number of migrations here
        ], 'migrations');
      }

      // publish the notifications
      $this->publishes([
        __DIR__ . '/../src/Notifications/ExpoPusherNotification.php' => app_path('Notifications/ExpoPusherNotification.php'),
      ], 'notification');
    }

    //register channel class
    $repository = $this->getInterestsDriver();
    $this->app->when(ExpoChannel::class)
      ->needs(Expo::class)
      ->give(function () use ($repository) {
        return new Expo(new ExpoRegistrar($repository));
      });

    //regiter routes
    $this->registerRoutes();
  }

  /**
   * Register routes
   *
   * @return void
   */
  protected function registerRoutes()
  {
    Route::group($this->routeConfiguration(), function () {
      $this->loadRoutesFrom(__DIR__ . '/../routes/web.php');
    });
  }

  /**
   * Default route configuration
   *
   * @return void
   */
  protected function routeConfiguration()
  {
    return [
      'prefix' => config('expo_pusher.route.prefix'),
      'middleware' => config('expo_pusher.route.middleware'),
      'as' => config('expo_pusher.route.name') . ".",
    ];
  }

  /**
   * Gets the Expo repository driver based on config.
   *
   * @return ExpoRepository
   */
  public function getInterestsDriver()
  {
    $driver = config('expo_pusher.drivers.name');

    switch ($driver) {
      case 'database':
        return new ExpoPusherDatabaseDriver();
      case 'redis':
        return new ExpoPusherRedisDriver(config('expo_pusher.drivers.redis.connection'));
      default:
        return new ExpoPusherFileDriver(config('expo_pusher.drivers.file.path'));
    }
  }
}
