<?php

namespace V360\ExpoPusher;

use ExponentPhpSDK\Expo;
use ExponentPhpSDK\Exceptions\ExpoException;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;
use Illuminate\Notifications\Events\NotificationFailed;
use V360\ExpoPusher\Repositories\ExpoPusherDatabaseDriver;
use V360\ExpoPusher\Repositories\ExpoPusherFileDriver;
use V360\ExpoPusher\Repositories\ExpoPusherRedisDriver;

class ExpoChannel
{

  /**
   * Expo SDK Object
   *
   * @var Expo
   */
  public $expo;


  /**
   * Constructor
   *
   * @param Expo $expo
   */
  public function __construct(Expo $expo)
  {
    //initialize
    $this->expo = $expo;

    //fetch access token
    $accessToken = config('expo_pusher.access_token');
    if ($accessToken) {
      // The access token will be attached to every push request you make hereafter
      $this->expo->setAccessToken($accessToken);
    }
  }

  /**
   * Send the given notification.
   *
   * @param  mixed  $notifiable
   * @param  \Illuminate\Notifications\Notification  $notification
   *
   * @throws NotificationFailed
   *
   * @return void
   */
  public function send($notifiable, Notification $notification)
  {
    //subscribe
    $interest = self::prepareInterest($notifiable);

    //prepare interests
    $interests = [$interest];

    try {
      $this->expo->notify(
        $interests,
        $notification->toExpoPush($notifiable)->toArray(),
        config('expo_pusher.debug')
      );
    } catch (ExpoException $th) {
      Log::error("ExpoNotifyException", [$th->getMessage()]);
      NotificationFailed::dispatch($notifiable, $notification, 'expo-pusher', [$th->getMessage()]);
    } catch (\Throwable $th) {
      Log::error("ExpoPusherException", [$th->getMessage()]);
      NotificationFailed::dispatch($notifiable, $notification, 'expo-pusher', [$th->getMessage()]);
    }
  }

  /**
   * Prepare interest channel for specified notifiable
   *
   * @param mixed $notifiable
   * @return string
   */
  public static function prepareInterest($notifiable)
  {
    $class = str_replace('\\', '.', get_class($notifiable));

    return $class . '.' . $notifiable->getKey();
  }

  /**
   * Notifiable is subscribed for notification or not
   *
   * @param mixed $notifiable
   * @return boolean
   */
  public static function isSubscribed($notifiable)
  {
    //prepare interest name
    $interest = self::prepareInterest($notifiable);
    switch (config('expo_pusher.drivers.name')) {
      case 'database':
        return ExpoPusherDatabaseDriver::exists($interest);
      case 'redis':
        return ExpoPusherRedisDriver::exists($interest);
      default:
        return ExpoPusherFileDriver::exists($interest);
    }
  }
}
