<?php

namespace V360\ExpoPusher;

class ExpoMessage
{

  protected $title;
  protected $body;
  protected $priority = 'default';
  protected $subtitle;
  protected $sound = "default";
  protected $badge = 0;
  protected $data;

  /**
   * Constructor
   */
  public function __construct(string $body = '')
  {
    $this->body = $body;
  }

  /**
   * Create a message with given body.
   *
   * @param  string  $body
   * @return static
   */
  public static function create($body = '')
  {
    return new static($body);
  }

  /**
   * Set title to display in the notification. Often displayed above the notification body
   * 
   * iOS and Android
   * 
   * @param string $body
   * @return ExpoMessage
   */
  public function title(string $title)
  {
    $this->title = $title;
    return $this;
  }

  /**
   * The message to display in the notification
   * 
   * iOS and Android
   *
   * @param string $body
   * @return $this
   */
  public function body(string $body)
  {
    $this->body = $body;
    return $this;
  }

  /**
   * The delivery priority of the message. 
   * Specify "default" or omit this field to use the default priority on each platform 
   * ("normal" on Android and "high" on iOS).
   * 
   * iOS and Android
   *
   * @param string $priority "default" | "normal" | "high"
   * @return $this
   */
  public function priority(string $priority)
  {
    $this->priority = $priority;
    return $this;
  }

  /**
   * The subtitle to display in the notification below the title.
   * 
   * iOS
   *
   * @param string $subtitle
   * @return $this
   */
  public function subtitle(string $subtitle)
  {
    $this->subtitle = $subtitle;
    return $this;
  }

  /**
   * Play a sound when the recipient receives this notification.
   * Specify "default" to play the device's default notification sound, or omit this field to play no sound. 
   * Custom sounds are not supported.
   * 
   * iOS
   *
   * @return $this
   */
  public function disableSound()
  {
    $this->sound = null;
    return $this;
  }

  /**
   * Number to display in the badge on the app icon. Specify zero to clear the badge.
   * iOS
   *
   * @param string $subtitle
   * @return $this
   */
  public function badge(string $badge)
  {
    $this->badge = $badge;
    return $this;
  }

  /**
   * A JSON object delivered to your app.
   * Maxsize 4KiB
   * 
   * iOS and Android
   *
   * @param string $subtitle
   * @return $this
   */
  public function data(array $data)
  {
    $this->data = $data;
    return $this;
  }

  /**
   * Get an array representation of the message.
   *
   * @return array
   */
  public function toArray()
  {
    $message = [
      'title'     =>  $this->title,
      'body'      =>  $this->body,
      'sound'     =>  $this->sound,
      'badge'     =>  $this->badge,
      'priority'  =>  $this->priority,
    ];
    if ($this->data) {
      $message['data'] = json_encode($this->data);
    }
    return $message;
  }
}
