<?php

namespace V360\DurationLogger;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use V360\DurationLogger\Enums\DurationFrequency;
use V360\DurationLogger\Enums\ScoreType;
use V360\DurationLogger\Models\DurationLogger;

class StatsDuration
{
  /**
   * Query builder used for this instance
   *
   * @var Builder
   */
  private $query;

  /**
   * Group by fields
   *
   * @var array
   */
  private $groupBy = [];

  private $durationType;
  private $scoreType;


  public function __construct()
  {
    $this->resetQuery();
  }

  public function resetQuery()
  {
    $this->query = DurationLogger::query();
    $this->scoreType = null;
    $this->durationType = null;
    return $this;
  }

  public function query()
  {
    return $this->resetQuery();
  }

  /**
   * Bind loggable with query
   *
   * @param string $loggable
   * @param int|null $loggableId
   * @return self
   */
  public function loggable($loggable, $loggableId = null)
  {
    $this->query->ofLoggable($loggable, $loggableId);
    return $this;
  }

  /**
   * Set duration frequency for query
   *
   * @param DurationFrequency|array $durationFrequency
   * @return self
   */
  public function frequency($durationFrequency)
  {
    $this->durationType = $durationFrequency;
    return $this;
  }

  /**
   * After the specified timestamp
   *
   * @param Carbon|string $start
   * @return self
   */
  public function after($start)
  {
    if (is_string($start)) {
      $start = Carbon::parse($start);
    }
    $this->query->durationAfter($start);
    return $this;
  }

  /**
   * Before the specified timestamp
   *
   * @param Carbon|string $start
   * @return self
   */
  public function before($end)
  {
    if (is_string($end)) {
      $end = Carbon::parse($end);
    }
    $this->query->durationBefore($end);
    return $this;
  }

  /**
   * Between the specified timestamps
   *
   * @param Carbon|string $start
   * @param Carbon|string $end
   * @return self
   */
  public function between($start, $end)
  {
    if (is_string($start)) {
      $start = Carbon::parse($start);
    }
    if (is_string($end)) {
      $end = Carbon::parse($end);
    }
    $this->query->durationBetween($start, $end);
    return $this;
  }

  /**
   * For current hour
   *
   * @return self
   */
  public function hour($sub = 0)
  {
    //set frequency of hour
    $this->frequency(DurationFrequency::HOUR);
    $this->between(Carbon::now()->subHours($sub)->startOfHour(), Carbon::now()->endOfHour());
    return $this;
  }

  /**
   * For current day
   *
   * @return self
   */
  public function day($sub = 0)
  {
    //set frequency of day
    $this->frequency(DurationFrequency::DAY);
    $this->between(Carbon::now()->subDays($sub)->startOfDay(), Carbon::now()->endOfDay());
    return $this;
  }

  /**
   * For current week
   *
   * @return self
   */
  public function week($sub = 0)
  {
    //set frequency of week
    $this->frequency(DurationFrequency::WEEK);
    $this->between(Carbon::now()->subWeeks($sub)->startOfWeek(), Carbon::now()->endOfWeek());
    return $this;
  }

  /**
   * For current month
   *
   * @return self
   */
  public function month($sub = 0)
  {
    //set frequency of month
    $this->frequency(DurationFrequency::MONTH);
    $this->between(Carbon::now()->subMonths($sub)->startOfMonth(), Carbon::now()->endOfMonth());
    return $this;
  }

  /**
   * For current year
   *
   * @return self
   */
  public function year($sub = 0)
  {
    //set frequency of year
    $this->frequency(DurationFrequency::YEAR);
    $this->between(Carbon::now()->subYear($sub)->startOfYear(), Carbon::now()->endOfYear());
    return $this;
  }

  public function thisHour()
  {
    return $this->hour(0);
  }
  public function today()
  {
    return $this->day(0);
  }
  public function thisWeek()
  {
    return $this->week(0);
  }
  public function thisMonth()
  {
    return $this->month(0);
  }
  public function thisYear()
  {
    return $this->year(0);
  }

  /**
   * For last N days
   * 
   * @param int $n
   * @param DurationFrequency $durationFrequency
   * @return self
   */
  public function lastNDurations($n, DurationFrequency $durationFrequency)
  {
    //set frequency of day
    switch ($durationFrequency) {
      case DurationFrequency::HOUR:
        return $this->hour($n);
      case DurationFrequency::DAY:
        return $this->day($n);
      case DurationFrequency::WEEK:
        return $this->week($n);
      case DurationFrequency::MONTH:
        return $this->month($n);
      case DurationFrequency::YEAR:
        return $this->year($n);
    }
  }

  /**
   * Records of specific score type
   *
   * @param ScoreType|array $scoreType Score type or list of it
   * @return self
   */
  public function score($scoreType)
  {
    $this->scoreType = $scoreType;
    if (is_array($scoreType)) {
      $this->byScoreType();
    }
    return $this;
  }

  /**
   * Group by loggable type
   *
   * @return self
   */
  public function byModel()
  {
    $this->groupBy[] = 'loggable_type';
    return $this;
  }

  /**
   * Group by loggable id
   *
   * @return self
   */
  public function byId()
  {
    $this->groupBy[] = 'loggable_id';
    return $this;
  }

  /**
   * Group by score type
   *
   * @return self
   */
  public function byScoreType()
  {
    $this->groupBy[] = 'score_type';
    return $this;
  }

  /**
   * Group by score value
   *
   * @return self
   */
  public function byScoreValues()
  {
    $this->groupBy[] = 'score_value';
    return $this;
  }

  /**
   * Count visits
   *
   * @return mixed
   */
  public function count()
  {
    $this->query->ofScoreType($this->scoreType);
    if ($this->durationType) {
      $this->query->ofDurationType($this->durationType);
    } else {
      $this->query->ofDurationType(DurationFrequency::YEAR);
    }
    if ($this->groupBy) {
      $groupBy = array_values(array_unique($this->groupBy));
      $this->query->groupBy($groupBy);
      $select = array_merge($groupBy, [DB::raw('CAST(SUM(score) AS UNSIGNED) AS total')]);
      $this->query->select($select);
      $results = $this->query->get();
    } else {
      $results = intval($this->query->sum('score'));
    }
    $this->resetQuery();
    return $results;
  }

  /**
   * Public function to return score values
   *
   * @param ScoreType $scoreType
   * @return Collection
   */
  public function scoreValues($scoreType = null)
  {
    $results = $this->query
      ->when($scoreType, function ($query) use ($scoreType) {
        $query->ofScoreType($scoreType);
      })
      ->distinct()
      ->select(['score_value'])
      ->get()
      ->pluck('score_value');

    $this->resetQuery();
    return $results;
  }

  public function deviceTypes()
  {
    $this->query->ofScoreType(ScoreType::DEVICE_TYPE);
    return $this->scoreValues();
  }

  public function browserFamilies()
  {
    $this->query->ofScoreType(ScoreType::BROWSER_FAMILY);
    return $this->scoreValues();
  }

  public function browserNames()
  {
    $this->query->ofScoreType(ScoreType::BROWSER_NAME);
    return $this->scoreValues();
  }

  public function platformFamilies()
  {
    $this->query->ofScoreType(ScoreType::PLATFORM_FAMILY);
    return $this->scoreValues();
  }

  public function platformNames()
  {
    $this->query->ofScoreType(ScoreType::PLATFORM_NAME);
    return $this->scoreValues();
  }
}
