<?php

namespace V360\DurationLogger\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use V360\DurationLogger\Enums\DurationFrequency;
use V360\DurationLogger\Enums\ScoreType;
use Illuminate\Database\Eloquent\Builder;

class DurationLogger extends Model
{
  use HasFactory;

  protected $table = "v360_duration_logs";
  public $timestamps = false;

  public const CACHE_KEY = "V360.DurationLogger.CachedRecords";

  // Disable Laravel's mass assignment protection
  protected $guarded = [
    'id'
  ];

  protected $casts = [
    'score_type' => ScoreType::class,
    'duration_type' => DurationFrequency::class,
    'duration_at' => 'datetime',
  ];

  /**
   * Get connectio nname via config
   *
   * @return string|null
   */
  public function getConnectionName()
  {
    return config('durationlogger.connection');
  }

  public function loggable()
  {
    return $this->morphTo();
  }

  /**
   * Scope of loggable
   *
   * @param Builder $query
   * @param string $loggableType
   * @param int|null $loggableId
   * @return void
   */
  public function scopeOfLoggable($query, $loggableType, $loggableId = null)
  {
    $query->where('loggable_type', $loggableType)
      ->when($loggableId, function ($query) use ($loggableId) {
        $query->where('loggable_id', $loggableId);
      });
  }

  public function scopeOfScoreType($query, $scoreType)
  {
    if (is_array($scoreType)) {
      $query->whereIn('score_type', $scoreType);
    } else {
      $query->where('score_type', $scoreType);
    }
  }
  public function scopeOfScoreValue($query, $scoreValue)
  {
    $query->where('score_value', $scoreValue);
  }

  /**
   * Duration type scopes
   */
  public function scopeHour($query)
  {
    $query->where('duration_type', DurationFrequency::HOUR);
  }
  public function scopeDay($query)
  {
    $query->where('duration_type', DurationFrequency::DAY);
  }
  public function scopeWeek($query)
  {
    $query->where('duration_type', DurationFrequency::WEEK);
  }
  public function scopeMonth($query)
  {
    $query->where('duration_type', DurationFrequency::MONTH);
  }
  public function scopeYear($query)
  {
    $query->where('duration_type', DurationFrequency::YEAR);
  }
  public function scopeOfDurationType($query, $durationType)
  {
    $query->where('duration_type', $durationType);
  }

  /**
   * Duration scope
   */
  public function scopeDurationAfter($query, $start, $op = '>=')
  {
    $query->where('duration_at', $op, $start);
  }
  public function scopeDurationBefore($query, $end, $op = '<=')
  {
    $query->where('duration_at', $op, $end);
  }
  public function scopeDurationBetween($query, $start, $end)
  {
    $query->whereBetween('duration_at', [$start, $end]);
  }
}
