<?php

namespace V360\DurationLogger;

use hisorange\BrowserDetect\Facade as Browser;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cache;
use V360\DurationLogger\Enums\DurationFrequency;
use V360\DurationLogger\Enums\ScoreType;
use V360\DurationLogger\Models\DurationLogger;

class LogDuration
{
  private $loggable;
  private $factor;

  public function __construct()
  {
  }

  /**
   * Increment by factor
   *
   * @param mixed $loggable
   * @param integer $factor
   * @return void
   */
  public function increment($loggable, $factor = 1)
  {
    $this->loggable = $loggable;
    $this->factor = $factor;
    return $this;
  }

  /**
   * Decrement by factor
   *
   * @param mixed $loggable
   * @param integer $factor
   * @return void
   */
  public function decrement($loggable, $factor = 1)
  {
    $this->loggable = $loggable;
    $this->factor = (-1) * $factor;
    return $this;
  }

  /**
   * Log records
   *
   * @return void
   */
  public function score()
  {
    //check for status, and in case status is off do not perform anything
    if (boolval(config('durationlogger.status')) == false) {
      return;
    }

    //detect params
    $scoreTypes = collect();
    foreach ($this->loggable->getScoreTypes() as $scoreType) {
      switch ($scoreType) {
        case ScoreType::DEVICE_TYPE:
          $scoreTypes->put(ScoreType::DEVICE_TYPE->value, Browser::deviceType());
          break;
        case ScoreType::BROWSER_NAME:
          $scoreTypes->put(ScoreType::BROWSER_NAME->value, Browser::browserName());
          break;
        case ScoreType::BROWSER_FAMILY:
          $scoreTypes->put(ScoreType::BROWSER_FAMILY->value, Browser::browserFamily());
          break;
        case ScoreType::PLATFORM_NAME:
          $scoreTypes->put(ScoreType::PLATFORM_NAME->value, Browser::platformName());
          break;
        case ScoreType::PLATFORM_FAMILY:
          $scoreTypes->put(ScoreType::PLATFORM_FAMILY->value, Browser::platformFamily());
          break;
      }
    }

    //consider frequency
    $now = Carbon::now();
    $durationTypes = collect();
    foreach ($this->loggable->getDurationFrequencies() as $duration) {
      switch ($duration) {
        case DurationFrequency::HOUR:
          $durationTypes->put(DurationFrequency::HOUR->value, $now->clone()->startOfHour());
          break;
        case DurationFrequency::DAY:
          $durationTypes->put(DurationFrequency::DAY->value, $now->clone()->startOfDay());
          break;
        case DurationFrequency::WEEK:
          $durationTypes->put(DurationFrequency::WEEK->value, $now->clone()->startOfWeek());
          break;
        case DurationFrequency::MONTH:
          $durationTypes->put(DurationFrequency::MONTH->value, $now->clone()->startOfMonth());
          break;
        case DurationFrequency::YEAR:
          $durationTypes->put(DurationFrequency::YEAR->value, $now->clone()->startOfYear());
          break;
      }
    }

    $records = collect();
    foreach ($durationTypes as $durationType => $durationAt) {
      //record score type spcific counts
      foreach ($scoreTypes as $scoreType => $scoreValue) {
        $records->push([
          'loggable_type' => $this->loggable->scoreKey ?? get_class($this->loggable),
          'loggable_id' => null,
          'score_type' => $scoreType,
          'score_value' => $scoreValue,
          'factor' => $this->factor,
          'duration_type' => $durationType,
          'duration_at' => $durationAt->toDateTimeString()
        ]);
        if ($this->loggable->getScoreWithDetail()) {
          $records->push([
            'loggable_type' => $this->loggable->scoreKey ?? get_class($this->loggable),
            'loggable_id' => $this->loggable->getKey(),
            'score_type' => $scoreType,
            'score_value' => $scoreValue,
            'factor' => $this->factor,
            'duration_type' => $durationType,
            'duration_at' => $durationAt->toDateTimeString()
          ]);
        }
      }
      //record score type independent count
      $records->push([
        'loggable_type' => $this->loggable->scoreKey ?? get_class($this->loggable),
        'loggable_id' => null,
        'factor' => $this->factor,
        'duration_type' => $durationType,
        'duration_at' => $durationAt->toDateTimeString()
      ]);
      if ($this->loggable->getScoreWithDetail()) {
        $records->push([
          'loggable_type' => $this->loggable->scoreKey ?? get_class($this->loggable),
          'loggable_id' => $this->loggable->getKey(),
          'factor' => $this->factor,
          'duration_type' => $durationType,
          'duration_at' => $durationAt->toDateTimeString()
        ]);
      }
    }

    //preserve records in cache
    $cachedRecords = Cache::get(DurationLogger::CACHE_KEY, []);
    $cacheKey = $now->startOfMinute()->timestamp;
    $keyRecords = $cachedRecords[$cacheKey] ?? [];
    $cachedKeyRecords = array_merge($keyRecords, $records->toArray());
    $cachedRecords[$cacheKey] = $cachedKeyRecords;
    Cache::put(DurationLogger::CACHE_KEY, $cachedRecords);
  }
}
