<?php

namespace V360\LivewireComponents\View\Livewire;

use Livewire\Component;
use Illuminate\Support\Str;

class AutoComplete extends Component
{

  public $query;
  public $autoKey;
  public $context;
  public $label;
  public $placeholder;
  public $value;
  public $allItems;

  public $isActive = false;
  public $allowNew = false;

  protected $listeners = [
    'AutoCompleteSelected' => 'onSelected',
    'AutoCompleteReset' => 'onReset',
    'AutoCompleteSelect' => 'onSelect',
    'AutoCompleteUpdateItems' => 'onUpdateItems'
  ];

  /**
   * Mount the component
   *
   * @param string $autoKey - Unique key by which this component will be identified
   * @param Collection $items - Items collection
   * @param string $label - Label to be used 
   * @param string $placeholder - Placeholder to be used 
   * @param mixed $value - Default value for the component
   * @param boolean $allowNew - New value is allowed
   * @return void
   */
  public function mount($autoKey, $items, $label = null, $placeholder = null, $value = null, $allowNew = false)
  {
    $this->autoKey = $autoKey;
    $this->allItems = $items;
    $this->label = $label;
    $this->context = (string) Str::ulid();
    $this->value = $value;
    $this->allowNew = $allowNew;
    $this->placeholder = $placeholder ?? "Select {$label}";
    $this->setDefaultValue();
  }

  /**
   * Clear query
   *
   * @return void
   */
  public function clear()
  {
    $this->setValue(null);
    $this->reset(['query', 'isActive']);
  }

  /**
   * Add new item
   *
   * @return void
   */
  public function add()
  {
    if ($this->allowNew) {
      $this->setValue($this->query);
    }
  }

  /**
   * Set default value, if supplied
   *
   * @return void
   */
  public function setDefaultValue()
  {
    if ($this->value) {
      $this->query = $this->allItems->filter(fn ($value, $key) => $key == $this->value)->first() ?? null;
    }
  }

  /**
   * Set value of component
   *
   * @param string $value
   * @return void
   */
  public function setValue($value)
  {
    $this->reset(['isActive']);
    if (method_exists($this, 'emitUp')) {
      $this->emitUp('AutoCompleteSet', $this->autoKey, $value);
    } else {
      $this->dispatch('AutoCompleteSet', field: $this->autoKey, value: $value);
    }
  }

  /**
   * Select item
   *
   * @param string $context
   * @param string $value
   * @param string $label
   * @return void
   */
  public function onSelected($context, $value, $label)
  {
    if ($this->context == $context) {
      $this->setValue($value);
      $this->query = $label;
    }
  }

  /**
   * Reset item
   *
   * @param string $autoKey
   * @return void
   */
  public function onReset($autoKey)
  {
    if ($this->autoKey == $autoKey) {
      $this->clear();
      $this->setDefaultValue();
    }
  }

  /**
   * Set item query
   *
   * @param string $autoKey
   * @return void
   */
  public function onSelect($autoKey, $value)
  {
    if ($this->autoKey == $autoKey) {
      $this->query = $value;
    }
  }

  /**
   * Reset items
   *
   * @param string $autoKey
   * @param Collection $items
   * @return void
   */
  public function onUpdateItems($autoKey, $items)
  {
    if ($this->autoKey == $autoKey) {
      $this->onReset($this->autoKey);
      $this->allItems = collect($items);
    }
  }

  /**
   * Make dropdown active once query is updated.
   *
   * @param string $value
   * @return void
   */
  public function updatedQuery($value)
  {
    $this->isActive = true;
  }

  /**
   * Filter items considering the query
   *
   * @return void
   */
  public function filterItems()
  {
    $items = $this->allItems
      ->when($this->query, function ($collection) {
        return $collection->filter(function ($value, $key) {
          return Str::contains($value, $this->query, true);
        });
      });
    return $items->sort();
  }

  /**
   * Render function
   *
   * @return void
   */
  public function render()
  {
    $items = $this->filterItems();
    if (method_exists($this, 'dispatchBrowserEvent')) {
      //v2
      $this->dispatchBrowserEvent('ActivateAutocomplete');
      return view('vlw::livewire.auto-complete', ['autoItems' => $items]);
    } else {
      //v3
      $this->dispatch('ActivateAutocomplete');
      return view('vlw::livewire.auto-complete-v3', ['autoItems' => $items]);
    }
  }
}
