<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * Password Laravel Blade Component
 *
 * A password input component with an optional show/hide toggle feature.
 * Features customizable labels and hints for enhanced user experience.
 *
 * Features:
 * - Show/hide password functionality
 * - Customizable labels and hints
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-07-15
 *
 * @example
 * <!-- Password input with show/hide toggle -->
 * <x-vform-password 
 *     name="user_password" 
 *     label="Password" 
 *     show="true" 
 *     hint="Enter your password." 
 * />
 *
 */
class Password extends Component
{

  public string $name;
  public ?string $value = null;
  public string $id;
  public ?string $label = null;
  public bool $show;
  public ?string $hint = null;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param string|null $value
   * @param string|null $id
   * @param mixed $label
   * @param bool $show
   * @param string|null $hint
   */
  public function __construct(
    string $name,
    ?string $value = null,
    ?string $id = null,
    mixed $label = null,
    bool $show = false,
    ?string $hint = null
  ) {
    $this->name = $name;
    $this->value = $value;
    $this->id = $id ? $id : str_replace('_', '-', $name);
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));
    $this->show = $show;
    $this->hint = $hint;
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.password');
  }
}
