<?php

namespace V360\FormComponents\View\Components;

use Closure;
use Illuminate\Contracts\View\View;
use Illuminate\View\Component;

/**
 * RangeSelector Laravel Blade Component
 *
 * A flexible range/slider component built on top of noUiSlider library.
 * Supports both single-value and dual-handle range selection with optional
 * predefined options (strings or numbers). Integrates seamlessly with
 * Livewire for reactive updates and Laravel's form validation.
 *
 * Features:
 * - Single value or range (dual-handle) selection
 * - Predefined options support (string or numeric)
 * - Optional numeric input fields for direct value entry (when options are null)
 * - Automatic pips generation based on data type
 * - Real-time value updates via Livewire
 * - Form validation integration
 * - Customizable styling with Bulma CSS framework
 * - Accessibility support with proper labels and hints
 *
 * CSS Customization:
 * The component supports CSS variables for visual customization with Bulma fallbacks:
 * - --range-selector-handle-background (fallback: --bulma-scheme-main)
 * - --range-selector-handle-shadow-inner (fallback: --bulma-scheme-main)
 * - --range-selector-handle-shadow-border (fallback: --bulma-border)
 * - --range-selector-handle-shadow-outer (fallback: --bulma-shadow)
 * - --range-selector-connect-background (fallback: --bulma-primary)
 * - --range-selector-connect-border (fallback: --bulma-border)
 * - --range-selector-pips-color (fallback: --bulma-text)
 *
 * Livewire Integration:
 * 
 * **IMPORTANT**: The component handles Livewire communication through custom events.
 * 
 * Events Dispatched:
 * - `RangeSelectorUpdated:{{ $name }}` - Fired when range/value selection changes (default behavior)
 *   Payload: { data: object } - Contains processed slider data (value/min/max/selectedOption)
 * - `RangeSelectorUpdated` - Fired when range/value selection changes (when dispatchGenericEvent=true)
 *   Payload: { name:string, data: object } - Contains processed slider data (value/min/max/selectedOption)
 * 
 * Events Listened For:
 * - `RangeSelectorReset` - Reset components to initial values
 *   Payload: { names: array<string> } - Array of component names to reset
 * - `RangeSelectorSet:{{ $name }}` - Set specific value(s) for this component
 *   Payload: { value: number|array } - Single value or [start, end] array for ranges
 *
 * @package V360\FormComponents\View\Components
 * @author Yogesh Kapuriya <yogesh@v360.tech>
 * @version 1.0.0
 * @since 2025-12-20
 *
 * @example
 * <!-- Single value selector -->
 * <x-vform-range-selector name="price" :min="0" :max="1000" :start="250" />
 *
 * @example
 * <!-- Range selector -->
 * <x-vform-range-selector name="price_range" :range="true" :min="0" :max="1000" :start="100" :end="800" />
 *
 * @example
 * <!-- Range selector with input fields -->
 * <x-vform-range-selector name="price_range" :range="true" :min="0" :max="1000" :start="100" :end="800" :show-input="true" />
 *
 * @example
 * <!-- Options-based selector -->
 * <x-vform-range-selector name="size" :options="['XS', 'S', 'M', 'L', 'XL']" :start="2" />
 *
 * @example
 * <!-- Advanced range configuration -->
 * <x-vform-range-selector 
 *     name="advanced_range" 
 *     :rangeConfig="['min' => 0, '10%' => 100, '50%' => 500, 'max' => 1000]" 
 *     :start="100" 
 *     label="Non-linear Range" 
 * />
 *
 * @example
 * <!-- Custom styling with CSS variables -->
 * <style>
 * :root {
 *   --range-selector-connect-background: #ff3860;
 *   --range-selector-handle-background: #ffffff;
 *   --range-selector-pips-color: #4a4a4a;
 * }
 * </style>
 * <x-vform-range-selector name="custom_range" :range="true" :min="0" :max="100" />
 *
 * @see https://refreshless.com/nouislider/ noUiSlider Documentation
 */
class RangeSelector extends Component
{
  /**
   * Unique identifier for the slider element
   *
   * @var string
   */
  public string $id;

  /**
   * JSON configuration for noUiSlider initialization
   *
   * @var string
   */
  public string $configJson = '';

  /**
   * JSON array of predefined options for selection
   *
   * @var string
   */
  public string $optionsJson = '[]';

  /**
   * Create a new RangeSelector component instance
   *
   * Initializes the range selector component with all necessary configuration.
   * When options are provided, the component automatically switches to options-based
   * mode where min/max/step are overridden to work with array indices.
   *
   * @param string $name Required. The form input name and component identifier
   * @param float $min Minimum selectable value (ignored when options provided)
   * @param float $max Maximum selectable value (ignored when options provided)
   * @param float $start Initial selected value or start of range
   * @param float $step Step increment for value changes (ignored when options provided)
   * @param bool $range Whether to enable dual-handle range selection
   * @param bool $showPips Whether to display pip markers along the slider track
   * @param bool $showInput Whether to show numeric input fields below the slider (only applies when options are null)
   * @param float $pipPositions Number of pip markers to display along the slider track (ignored when options provided)
   * @param array<int, string|int|float>|null $options Predefined options for selection (strings or numbers)
   * @param array<string, mixed>|null $rangeConfig Additional noUiSlider range options (see noUiSlider docs)
   * @param float|null $end End value for range selection (only used when range=true)
   * @param string|null $label Display label (auto-generated from name if null)
   * @param string|null $hint Help text shown below the slider
   * @param string|null $class Additional CSS classes to apply
   * @param bool $dispatchGenericEvent Whether to dispatch generic event (RangeSelectorUpdated) instead of specific (RangeSelectorUpdated:name)
   *
   * @throws \InvalidArgumentException When name is empty
   */
  public function __construct(
    public string $name,
    public float $min = 0,
    public float $max = 100,
    public float $start = 0,
    public float $step = 1,
    public bool $range = false,
    public bool $showPips = true,
    public bool $showInput = false,
    public float $pipPositions = 10,
    public ?array $options = null,
    public ?array $rangeConfig = null,
    public ?float $end = null,
    public ?string $label = null,
    public ?string $hint = null,
    public ?string $class = null,
    public bool $dispatchGenericEvent = false,
  ) {
    $this->label = $label ?? str($name)->headline()->toString();

    // Generate a unique ID based on the name
    $this->id = str($this->name)
      ->prepend('rs_')
      ->snake()
      ->replace('.', '_')
      ->toString();

    // Handle options-based configuration
    if (!empty($options)) {
      $this->configureForOptions();
      $this->showInput = false; // disable input fields when options are provided
    }

    // Set end value for range sliders
    if ($this->range) {
      $this->end ??= $this->max;
      $this->end = $this->validateEndValue($this->end);
    }

    // Validate start value
    $this->start = $this->validateStartValue($this->start);

    // Generate JSON configurations
    $this->configJson = $this->getSliderConfig();

    // Prepare options JSON
    $this->optionsJson = $this->options ? json_encode($this->options) : '[]';
  }

  /**
   * Get the view / contents that represent the component
   *
   * Returns the Blade template for rendering the range selector component.
   * The template includes the HTML structure, styling, and JavaScript
   * initialization for the noUiSlider.
   *
   * @return View|Closure|string The component view instance
   *
   */
  public function render(): View|Closure|string
  {
    return view('vform::components.range-selector');
  }

  /**
   * Configure slider for options-based mode
   *
   * When options array is provided, this method reconfigures the slider
   * to work with array indices instead of the original numeric range.
   * This allows the slider to work with string options like sizes, categories, etc.
   *
   * Changes made:
   * - Sets min to 0 (first option index)
   * - Sets max to count of options - 1 (last option index)
   * - Forces step to 1 (move by one option at a time)
   *
   * @return void
   *
   * @internal This method is called automatically during construction
   */
  private function configureForOptions(): void
  {
    $this->min = 0;
    $this->max = \count($this->options) - 1;
    $this->step = 1;
  }

  /**
   * Validate start value within acceptable bounds
   *
   * Ensures the start value falls within the defined min/max range.
   * If the provided value is outside bounds, it will be clamped to
   * the nearest valid value.
   *
   * @param float $start The proposed start value
   *
   * @return float The validated start value within bounds
   *
   * @internal This method is called automatically during construction
   */
  private function validateStartValue(float $start): float
  {
    return max($this->min, min($this->max, $start));
  }

  /**
   * Validate end value within acceptable bounds
   *
   * Ensures the end value falls within the defined range and is not
   * less than the start value. If the provided value is invalid,
   * it will be clamped to the nearest valid value.
   *
   * @param float $end The proposed end value
   *
   * @return float The validated end value within bounds and >= start value
   *
   * @internal This method is called automatically during construction when range=true
   */
  private function validateEndValue(float $end): float
  {
    return max($this->start, min($this->max, $end));
  }

  /**
   * Generate noUiSlider configuration as JSON string
   *
   * Creates the complete configuration object required by noUiSlider library
   * for initialization. The configuration includes start values, connection
   * settings, range definitions, step increments, and pip markers.
   *
   * Configuration structure:
   * - start: Initial value(s) - array for range, single value otherwise
   * - connect: Whether to show connection line between handles
   * - range: Object with min/max boundary definitions
   * - step: Increment value for slider movement
   * - pips: Marker configuration based on pipPositions (only when no options provided)
   *
   * When options are provided, pips are automatically generated for each option.
   * When no options are provided, pips are distributed evenly based on pipPositions.
   *
   * @return string JSON-encoded configuration object for noUiSlider
   *
   * @throws \JsonException If JSON encoding fails
   *
   * @see https://refreshless.com/nouislider/slider-options/ noUiSlider Configuration Options
   */
  public function getSliderConfig(): string
  {
    // prepare range configuration
    $range = $this->rangeConfig ?? [
      'min' => $this->min,
      'max' => $this->max
    ];
    $config = [
      'start' => $this->range ? [$this->start, $this->end] : [$this->start],
      'connect' => $this->range,
      'range' => $range,
      'step' => $this->step,
    ];
    // generate pips only if no options are provided
    if ($this->showPips && empty($this->options)) {
      if ($this->rangeConfig) {
        $pips = [
          'mode' => 'range',
          'density' => 2,
        ];
      } else {
        $range = $this->max - $this->min;
        $this->pipPositions = $range < $this->pipPositions ? $range : $this->pipPositions;
        $pips = [
          'mode' => 'positions',
          'values' => range(0, 100, 100 / $this->pipPositions),
          'density' => 3,
          'stepped' => true,
        ];
      }
      $config['pips'] = $pips;
    }
    return json_encode($config);
  }
}