<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * Autocomplete Laravel Blade Component
 *
 * An autocomplete input component that provides dynamic suggestions as users type.
 * Supports customizable labels, hints, and placeholder text for enhanced user experience.
 * Can be integrated with Livewire for real-time updates.
 *
 * Features:
 * - Dynamic suggestions based on user input
 * - Customizable labels and hints
 * - Placeholder text support
 * - Option to search by start of the string or anywhere
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 * 
 * Livewire Integration:
 * 
 * **IMPORTANT**: The component handles Livewire communication through custom events.
 * 
 * Events Dispatched:
 * - `AutoCompleteSet:{{ $name }}` - Fired when value is set
 *   Payload: { value: string }
 * 
 * Events Listened For:
 * - `AutoCompleteInit:{{ $name }}` - Initialize component with value and label
 *   Payload: { value: string, label: string }
 * - `AutoCompleteReset` - Reset component for specified fields
 *   Payload: { autoFields: array }
 * - `AutoCompleteItems:{{ $name }}` - Update available items/options
 *   Payload: { items: array }
 * 
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-06-25
 * 
 * 
 *
 * @example
 * <!-- Autocomplete input with options -->
 * <x-vform-autocomplete 
 *     name="city" 
 *     :options="['New York', 'Los Angeles', 'Chicago', 'Houston', 'Phoenix']" 
 *     label="City" 
 *     placeholder="Start typing a city..." 
 *     hint="Select your city from the suggestions." 
 * />
 *
 */
class Autocomplete extends Component
{
  /**
   * Create a new component instance
   *
   * @param string $name Name of the input
   * @param mixed $options Options for the autocomplete Array or Collection
   * @param string|null $value Value of the input
   * @param string|null $id ID of the input, in case of not provided, slug version of the name will be used
   * @param string|null $label Label of the input, in case of not provided, title version of the name will be used
   * @param string|null $placeholder Placeholder of the input, default is "Select"
   * @param string|null $hint Hint of the input
   * @param boolean $startSearch Search by start or not, default is false
   * @param boolean $isLivewire Livewire component or not, default is false
   */
  public function __construct(
    public string $name,
    public mixed $options,
    public ?string $value = null,
    public ?string $id = null,
    public ?string $label = null,
    public ?string $placeholder = null,
    public ?string $hint = null,
    public bool $startSearch = false,
    public bool $isLivewire = false,
  ) {
    $this->id = $id ? $id : str_replace('_', '-', $name);
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));
    $this->placeholder = $placeholder ? $placeholder : "Select";
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.autocomplete');
  }
}
