<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * Text Laravel Blade Component
 *
 * A text input component with customizable labels, icons, hints, and addons.
 * Features enhanced user experience with optional icons and addons.
 *
 * Features:
 * - Customizable labels and hints
 * - Optional icons for better UX
 * - Addon support for additional context
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-07-05
 *
 * @example
 * <!-- Basic text input -->
 * <x-vform-text 
 *     name="username" 
 *     label="Username" 
 *     hint="Enter your username." 
 * />
 *
 * <!-- Text input with icon and addon -->
 * <x-vform-text 
 *     name="website" 
 *     label="Website" 
 *     icon="link" 
 *     addon=".com" 
 *     hint="Enter your website." 
 * />
 *
 */
class Text extends Component
{

  public string $name;
  public ?string $value = null;
  public string $id;
  public ?string $label = null;
  public string $icon;
  public string $class;
  public ?string $hint = null;
  public ?string $placeholder = null;
  public ?string $addon = null;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param mixed $value
   * @param string|null $id
   * @param mixed $label
   * @param mixed $icon - true, false, 'text_fields'
   * @param string|null $placeholder
   * @param string|null $classes
   * @param string|null $hint
   * @param string|null $addon
   */
  public function __construct(
    string $name,
    mixed $value = null,
    ?string $id = null,
    mixed $label = null,
    mixed $icon = 'text_fields',
    ?string $placeholder = null,
    ?string $classes = null,
    ?string $hint = null,
    ?string $addon = null
  ) {
    $this->name = $name;
    $this->value = $value;
    $this->id = $id ? $id : str_replace('_', '-', $name);
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));
    $this->icon = $icon;
    $this->class = "input $classes";
    if ($this->label === false) {
      $this->placeholder = Str::title(str_replace('_', ' ', $name));
    }
    if ($placeholder) {
      $this->placeholder = $placeholder;
    }
    $this->hint = $hint;
    $this->addon = $addon;
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.text');
  }
}
