<?php

namespace V360\FormComponents\View\Components;

use Closure;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * InputTags Laravel Blade Component
 *
 * An input tags component that allows users to select multiple tags from a predefined list.
 * Features customizable labels and supports both array and key-value pair options.
 *
 * Features:
 * - Multiple tag selection
 * - Customizable labels
 * - Supports array and key-value pair options
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * Livewire Integration:
 * 
 * **IMPORTANT**: The component handles Livewire communication through custom events.
 * 
 * Events Dispatched:
 * - `InputTagsChanged` - Fired when tag selection changes
 *   Payload: { value: array }
 * 
 * Events Listened For:
 * - `InitInputTags:{{ $name }}` - Initialize component with initial value
 *   Payload: { value: array }
 * - `ResetInputTags:{{ $name }}` - Reset component to empty state
 *   Payload: none
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-07-15
 *
 * @example
 * <!-- Input tags with predefined options -->
 * <x-vform-input-tags 
 *     name="tags" 
 *     :options="['Laravel', 'VueJS', 'Livewire']" 
 *     label="Select Tags" 
 * />
 *
 * <!-- Input tags with key-value pair options -->
 * <x-vform-input-tags 
 *     name="technologies" 
 *     :options="['laravel' => 'Laravel', 'vuejs' => 'VueJS', 'livewire' => 'Livewire']" 
 *     :key-value="true" 
 *     label="Select Technologies" 
 * />
 *
 */
class InputTags extends Component
{
  /**
   * Constructor
   * 
   * @param string $name The name of the input field
   * @param mixed|null $label The label for the input field
   * @param mixed $id The unique id of the input field
   * @param mixed $value The initial selected tags
   * @param array|\Illuminate\Support\Collection $options Available tag options
   * @param bool $keyValue Whether options are key-value pairs
   */
  public function __construct(
    public string $name,
    public mixed $label = null,
    public ?string $id = null,
    public mixed $value = [],
    public array|Collection $options = [],
    public bool $keyValue = false,
  ) {
    //
    $this->id = $id ?? Str::uuid();
    $this->label = $label !== false ? ($label ?? Str::headline($name)) : false;
    if (is_string($value)) {
      $this->value = json_decode($value);
    } else if (!is_array($value)) {
      $this->value = $this->value->toArray();
    }
  }

  /**
   * Get the view / contents that represent the component.
   */
  public function render(): View|Closure|string
  {
    return view('vform::components.input-tags');
  }
}
