/**
 * SelectCheck - A class for managing VirtualSelect instances with Livewire integration
 */
export default class SelectCheck {
  constructor(elementId, config = {}, isMultiple = false) {
    this.elementId = elementId;
    this.config = config;
    this.isMultiple = isMultiple;
    this.isInitializing = true;
    this.virtualSelectElement = null;
    this.debug = config.debug || false;

    this.init();
  }

  /**
   * Log debug messages if debug mode is enabled
   */
  logDebug(...args) {
    if (this.debug) {
      console.debug(...args);
    }
  }

  /**
   * Initialize the SelectCheck component
   */
  init() {
    const selectElement = document.getElementById(this.elementId);

    if (!selectElement) {
      console.warn(
        `SelectCheck: Element with ID '${this.elementId}' not found`
      );
      return;
    }

    if (typeof window.VirtualSelect === "undefined") {
      console.warn("SelectCheck: VirtualSelect library not found");
      return;
    }

    // Initialize VirtualSelect with the configuration
    window.VirtualSelect.init({
      ele: selectElement,
      ...this.config,
    });

    this.virtualSelectElement = document.getElementById(this.elementId);

    if (this.virtualSelectElement) {
      this.bindEvents();
      this.setupLivewireListeners();

      // Mark initialization as complete after a short delay
      setTimeout(() => {
        this.isInitializing = false;
      }, 100);
    }
  }

  /**
   * Bind event listeners to the VirtualSelect element
   */
  bindEvents() {
    if (!this.virtualSelectElement) return;

    this.virtualSelectElement.addEventListener("change", (event) => {
      this.handleSelectionChange(event);
    });
  }

  /**
   * Handle selection change events
   */
  handleSelectionChange(event) {
    // Skip dispatching during initialization
    if (this.isInitializing) return;

    const selectedValues = event.target.value;

    this.logDebug("SelectCheck Selection Changed:", {
      id: this.elementId,
      values: selectedValues,
      isMultiple: this.isMultiple,
    });

    if (window.Livewire) {
      window.Livewire.dispatch(`SelectCheckUpdated:${this.elementId}`, {
        value: selectedValues,
      });
    }
  }

  /**
   * Setup Livewire event listeners
   */
  setupLivewireListeners() {
    if (!window.Livewire) return;

    // Reset all selections on Livewire event
    Livewire.on(`SelectCheckReset:${this.elementId}`, () => {
      this.reset();
    });

    // Remove specific values from selection on Livewire event
    Livewire.on(`SelectCheckRemoveValue:${this.elementId}`, ({ value }) => {
      this.removeValue(value);
    });
  }

  /**
   * Reset the selection
   */
  reset() {
    this.logDebug(`SelectCheck Reset triggered for ${this.elementId}`);

    if (this.virtualSelectElement && this.virtualSelectElement.reset) {
      this.virtualSelectElement.reset();
    }
  }

  /**
   * Remove specific values from the selection
   */
  removeValue(value) {
    this.logDebug(
      `SelectCheck RemoveValue triggered for ${this.elementId}:`,
      value
    );

    if (!value || !this.virtualSelectElement) return;

    const currentValues = this.virtualSelectElement.value;
    if (!currentValues) return;

    // Normalize both current values and incoming value to arrays for consistent processing
    const currentArray = Array.isArray(currentValues)
      ? currentValues
      : [currentValues];
    const valueArray = Array.isArray(value) ? value : [value];

    // Filter out the values to be removed
    const newValues = currentArray.filter((val) => !valueArray.includes(val));

    // Set the new values based on selection mode
    try {
      if (this.isMultiple) {
        // Multiple select - set as array
        this.virtualSelectElement.setValue(newValues);
      } else {
        // Single select - set as single value or empty
        this.virtualSelectElement.setValue(
          newValues.length > 0 ? newValues[0] : ""
        );
      }
    } catch (error) {
      console.error("Error setting SelectCheck values:", error);
    }
  }

  /**
   * Set the value of the SelectCheck
   */
  setValue(value) {
    if (!this.virtualSelectElement) return;

    try {
      this.virtualSelectElement.setValue(value);
    } catch (error) {
      console.error("Error setting SelectCheck value:", error);
    }
  }

  /**
   * Get the current value of the SelectCheck
   */
  getValue() {
    if (!this.virtualSelectElement) return null;
    return this.virtualSelectElement.value;
  }

  /**
   * Destroy the SelectCheck instance
   */
  destroy() {
    if (this.virtualSelectElement) {
      // Remove event listeners if needed
      this.virtualSelectElement.removeEventListener(
        "change",
        this.handleSelectionChange
      );

      // Destroy VirtualSelect instance if it has a destroy method
      if (this.virtualSelectElement.destroy) {
        this.virtualSelectElement.destroy();
      }
    }
  }

  /**
   * Static method to create and initialize a SelectCheck instance
   */
  static create(elementId, config = {}, isMultiple = false) {
    return new SelectCheck(elementId, config, isMultiple);
  }
}
