# Form Components based on Bulma

A package for components based on Laravel and Bulma. This package provides a comprehensive set of form components with built-in validation, styling, and enhanced functionality.

## Overview

- Built on Laravel's component system
- Integrates with Bulma CSS framework
- Includes advanced features like file uploads, date picking, and dynamic selects
- Supports form validation and custom rules
- Responsive and accessible form elements

## Dependencies

- [Laravel HTML](https://spatie.be/docs/laravel-html/v3/introduction)
- [jQuery](https://jquery.com/)
- [Bulma Calender](https://bulma-calendar.onrender.com/)
- [Bulma TagsInput](https://bulma-tagsinput.netlify.app/)
- [Cropper.js](https://github.com/fengyuanchen/cropperjs) For ImagePicker
- [WebcamJS](https://github.com/jhuckaby/webcamjs/) - For ImagePicker
- [flatpickr](https://flatpickr.js.org/) - For datetime picker
- [Filepond](https://pqina.nl/filepond/) - For File upload
- [Quill](https://quilljs.com/docs/quickstart) - For Rich text editor
- [noUlSlider](https://refreshless.com/nouislider/) - For RangeSelector
- [Choices.js](https://choices-js.github.io/Choices/) - For SelectChoice
- [VirtualSelect](https://sa-si-dev.github.io/virtual-select/) - For SelectCheck

## Installation

```bash
$ composer require v360-tech/components-form
```

In case your Laravel version does NOT autoload the packages, add the service provider to config/app.php:

```php
V360\FormComponents\FormComponentsServiceProvider::class,
```

`For image picker, Cropper object should be available on client side. So need to follow installation steps as per auther's document.`

## Export

### To publish views

```bash
php artisan vendor:publish --provider="V360\FormComponents\FormComponentsServiceProvider" --tag="views"
```

### To publish configs

`Required in case of ReCaptcha component.`

```bash
php artisan vendor:publish --provider="V360\FormComponents\FormComponentsServiceProvider" --tag="config"
```

### To publish assets

`Required for components that use JavaScript/CSS assets like RangeSelector, SelectCheck etc.`

```bash
php artisan vendor:publish --provider="V360\FormComponents\FormComponentsServiceProvider" --tag="assets"
```

## Component Documentation

This package implements a comprehensive set of form components. Each component is designed to be easily integrated into your Laravel application with minimal configuration.

### Basic Input Components

| Component | Tag                    | Description                 | Key Features                                              |
| --------- | ---------------------- | --------------------------- | --------------------------------------------------------- |
| Text      | `<x-vform-text />`     | Basic text input field      | - Label support<br>- Placeholder text<br>- Custom classes |
| Number    | `<x-vform-number />`   | Numeric input field         | - Min/max values<br>- Step control<br>- Right alignment   |
| Email     | `<x-vform-email />`    | Email input with validation | - Built-in email validation<br>- Custom error messages    |
| Password  | `<x-vform-password />` | Secure password input       | - Password strength indicator<br>- Show/hide toggle       |
| Textarea  | `<x-vform-textarea />` | Multi-line text input       | - Resizable<br>- Auto-grow option                         |

### Advanced Input Components

| Component      | Tag                          | Description            | Key Features                                                                                                                                                                                        |
| -------------- | ---------------------------- | ---------------------- | --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| Amount         | `<x-vform-amount />`         | Currency input field   | - Multiple currencies (₹,$,€,£)<br>- Right alignment<br>- Automatic formatting                                                                                                                      |
| Datepicker     | `<x-vform-datepicker />`     | Date/time selection    | - Date/Time/DateTime modes<br>- Min/max date limits<br>- Custom time restrictions                                                                                                                   |
| Autocomplete   | `<x-vform-autocomplete />`   | Input with suggestions | - Dynamic suggestions<br>- Custom data source<br>- Debounced search                                                                                                                                 |
| Dynamic Select | `<x-vform-dynamic-select />` | AJAX-powered select    | - Remote data loading<br>- Search functionality<br>- Pagination support                                                                                                                             |
| Tags           | `<x-vform-tags />`           | Tag input field        | - Multiple tags support<br>- Custom validation<br>- Auto-completion                                                                                                                                 |
| RangeSelector  | `<x-vform-range-selector />` | Range/slider input     | - Single or dual-handle sliders<br>- Predefined options support<br>- Optional numeric input fields<br>- Auto-generated pips<br>- CSS customization<br>- Livewire events<br>- noUiSlider integration |

### File and Media Components

| Component    | Tag                        | Description               | Key Features                                                  |
| ------------ | -------------------------- | ------------------------- | ------------------------------------------------------------- |
| File         | `<x-vform-file />`         | Basic file upload         | - Multiple file support<br>- File type validation             |
| FilePond     | `<x-vform-file-pond />`    | Advanced file upload      | - Drag & drop<br>- Image preview<br>- Upload progress         |
| Image Picker | `<x-vform-image-picker />` | Image selection/capture   | - Camera integration<br>- Image cropping<br>- Preview support |
| Media Select | `<x-vform-media-select />` | Media library integration | - Gallery view<br>- Multiple selection<br>- Filterable        |

### Selection Components

| Component    | Tag                         | Description                          | Key Features                                                                                                         |
| ------------ | --------------------------- | ------------------------------------ | -------------------------------------------------------------------------------------------------------------------- |
| Checkbox     | `<x-vform-checkbox />`      | Single/multiple checkboxes           | - Group support<br>- Custom styling                                                                                  |
| Radio        | `<x-vform-radio />`         | Radio button group                   | - Horizontal/Vertical layout<br>- Custom labels                                                                      |
| Select       | `<x-vform-select />`        | Dropdown selection                   | - Option groups<br>- Multiple selection<br>- Search filter                                                           |
| SelectChoice | `<x-vform-select-choice />` | Enhanced dropdown with Choices.js    | - Choices.js integration<br>- Advanced search<br>- Remove buttons<br>- Multiple selection limits                     |
| SelectCheck  | `<x-vform-select-check />`  | Enhanced dropdown with VirtualSelect | - VirtualSelect.js integration<br>- Search functionality<br>- Livewire events<br>- Maximum item limits               |
| SelectGrid   | `<x-vform-select-grid />`   | Visual grid-based selection          | - Grid layout with images/colors<br>- Single/multiple selection<br>- Flexible rows/columns<br>- Color swatch support |
| InputSwitch  | `<x-vform-input-switch />`  | Toggle switch control                | - Modern toggle UI<br>- Livewire support<br>- Custom colors                                                          |

### Special Components

| Component | Tag                      | Description                  | Key Features                                                     |
| --------- | ------------------------ | ---------------------------- | ---------------------------------------------------------------- |
| ReCaptcha | `<x-vform-re-captcha />` | Google ReCaptcha integration | - v2/v3 support<br>- Custom validation<br>- Error handling       |
| Table     | `<x-vform-table />`      | Dynamic table input          | - Add/Remove rows<br>- Column configuration<br>- Data validation |
| Keyvalue  | `<x-vform-keyvalue />`   | Key-value pair input         | - Dynamic pairs<br>- Custom validation<br>- Sortable entries     |

### Other components

| Element       | Tag                       |
| :------------ | ------------------------- |
| A Date Picker | \<x-vform-a-datepicker /> |

## Component Usage Examples

### Basic Components

```blade
{{-- Text Input --}}
<x-vform-text
    name="username"
    label="Username"
    placeholder="Enter username"
    :hint="'Must be unique'"
/>

{{-- Amount Input --}}
<x-vform-amount
    name="price"
    currency="dollar"
    :value="19.99"
    label="Price"
/>

{{-- Date Picker --}}
<x-vform-datepicker
    name="event_date"
    type="datetime"
    :dateLimits="['min' => now(), 'max' => now()->addYear()]"
    label="Event Date"
/>
```

### Advanced Components

```blade
{{-- Dynamic Select --}}
<x-vform-dynamic-select
    name="category"
    :url="route('api.categories')"
    label="Category"
    placeholder="Select a category"
/>

{{-- File Upload with FilePond --}}
<x-vform-file-pond
    name="documents[]"
    multiple
    accept="application/pdf"
    :maxFileSize="5000000"
    label="Upload Documents"
/>

{{-- Key-Value Pairs --}}
<x-vform-keyvalue
    name="meta_data"
    :value="$existingData"
    label="Meta Data"
/>

{{-- Toggle Switch --}}
<x-vform-input-switch
    name="is_active"
    label="Status"
    hint="Toggle to activate/deactivate"
/>

{{-- Range Selector - Basic Range --}}
<x-vform-range-selector
    name="price_range"
    :range="true"
    :min="0"
    :max="1000"
    :start="100"
    :end="800"
    label="Price Range"
    hint="Select your budget range"
/>

{{-- Range Selector - Single Value with Input Field --}}
<x-vform-range-selector
    name="discount"
    :range="false"
    :min="0"
    :max="50"
    :start="15"
    :step="5"
    :showInput="true"
    label="Discount Percentage"
/>

{{-- Range Selector - Options-based (Size Selection) --}}
<x-vform-range-selector
    name="size"
    :options="['XS', 'S', 'M', 'L', 'XL', 'XXL']"
    :start="2"
    :showPips="true"
    label="Select Size"
/>

{{-- Range Selector - Custom CSS Styling --}}
<style>
:root {
  --range-selector-connect-background: #ff3860;
  --range-selector-handle-background: #ffffff;
  --range-selector-pips-color: #4a4a4a;
}
</style>
<x-vform-range-selector
    name="temperature"
    :range="true"
    :min="-10"
    :max="40"
    :start="18"
    :end="25"
    :step="0.5"
    label="Temperature Range (°C)"
    class="custom-range"
/>

{{-- Range Selector - Options with Range Selection --}}
<x-vform-range-selector
    name="experience_range"
    :options="['Beginner', 'Intermediate', 'Advanced', 'Expert', 'Master']"
    :range="true"
    :start="1"
    :end="3"
    label="Experience Level Range"
    hint="Select minimum and maximum experience levels"
/>

{{-- Range Selector - Livewire Integration --}}
<x-vform-range-selector
    name="budget_range"
    :range="true"
    :min="0"
    :max="5000"
    :start="1000"
    :end="3000"
    label="Budget Range"
    wire:model="budgetRange"
    hint="Automatically updates Livewire property on change"
/>

{{-- Range Selector - Advanced Range Configuration --}}
<x-vform-range-selector
    name="non_linear_range"
    :rangeConfig="['min' => 0, '10%' => 100, '50%' => 500, 'max' => 1000]"
    :range="true"
    :start="100"
    :end="500"
    label="Non-linear Distribution"
    hint="Uses custom range configuration for non-linear value distribution"
/>

{{-- Enhanced Select with Choices.js --}}
<x-vform-select-choice
    name="categories"
    :options="$categoryOptions"
    :multiple="true"
    :maxItemCount="3"
    placeholder="Select up to 3 categories"
    label="Categories"
/>

{{-- Single Select with Search --}}
<x-vform-select-choice
    name="country"
    :options="['US' => 'United States', 'CA' => 'Canada', 'UK' => 'United Kingdom']"
    placeholder="Choose country"
    label="Country"
/>

{{-- Grouped Options --}}
<x-vform-select-choice
    name="product"
    :options="[
        'Fruits' => ['apple' => 'Apple', 'banana' => 'Banana'],
        'Vegetables' => ['carrot' => 'Carrot', 'broccoli' => 'Broccoli']
    ]"
    label="Product Category"
/>

{{-- VirtualSelect Dropdown - Multiple Selection --}}
<x-vform-select-check
    name="technologies"
    :options="$techOptions"
    :multiple="true"
    :maxItemCount="5"
    placeholder="Select up to 5 technologies"
    label="Technologies"
/>

{{-- VirtualSelect Dropdown - Single Selection --}}
<x-vform-select-check
    name="department"
    :options="['HR' => 'Human Resources', 'IT' => 'Information Technology', 'Finance' => 'Finance']"
    :multiple="false"
    placeholder="Choose department"
    label="Department"
/>

{{-- VirtualSelect with Grouped Options --}}
<x-vform-select-check
    name="course"
    :options="[
        'Programming' => ['php' => 'PHP', 'javascript' => 'JavaScript'],
        'Design' => ['ui' => 'UI Design', 'ux' => 'UX Design']
    ]"
    :multiple="true"
    label="Course Selection"
/>

{{-- Grid Selection - Color Theme Selection --}}
<x-vform-select-grid
    name="theme_color"
    :options="[
        ['value' => 'blue', 'label' => 'Ocean Blue', 'color' => '#3498db'],
        ['value' => 'green', 'label' => 'Forest Green', 'color' => '#2ecc71'],
        ['value' => 'red', 'label' => 'Crimson Red', 'color' => '#e74c3c'],
        ['value' => 'purple', 'label' => 'Royal Purple', 'color' => '#9b59b6']
    ]"
    :multiple="false"
    :optionWidth="120"
    :gridCols="4"
    label="Choose Theme Color"
/>

{{-- Grid Selection - Avatar/Image Selection --}}
<x-vform-select-grid
    name="avatar"
    :options="[
        ['value' => 'avatar1', 'label' => 'Professional', 'url' => '/images/avatars/pro.png'],
        ['value' => 'avatar2', 'label' => 'Casual', 'url' => '/images/avatars/casual.png'],
        ['value' => 'avatar3', 'label' => 'Creative', 'url' => '/images/avatars/creative.png']
    ]"
    :multiple="false"
    :optionWidth="120"
    :gridRows="1"
    label="Choose Avatar"
/>

{{-- Grid Selection - Mixed Content (Images + Colors) --}}
<x-vform-select-grid
    name="features"
    :options="[
        ['value' => 'premium', 'label' => 'Premium', 'color' => '#f39c12'],
        ['value' => 'wifi', 'label' => 'WiFi', 'url' => '/icons/wifi.svg'],
        ['value' => 'custom', 'label' => 'Custom Theme', 'color' => '#9b59b6'],
        ['value' => 'pool', 'label' => 'Pool', 'url' => '/icons/pool.svg']
    ]"
    :multiple="true"
    :optionWidth="150"
    :gridCols="2"
    label="Select Features"
    hint="Choose features for your property"
/>

{{-- Grid Selection - Flexible Layout using Rows --}}
<x-vform-select-grid
    name="sizes"
    :options="[
        ['value' => 'xs', 'label' => 'Extra Small', 'color' => '#95a5a6'],
        ['value' => 's', 'label' => 'Small', 'color' => '#3498db'],
        ['value' => 'm', 'label' => 'Medium', 'color' => '#f39c12'],
        ['value' => 'l', 'label' => 'Large', 'color' => '#e67e22'],
        ['value' => 'xl', 'label' => 'Extra Large', 'color' => '#e74c3c'],
        ['value' => 'xxl', 'label' => 'XXL', 'color' => '#8e44ad']
    ]"
    :multiple="false"
    :optionWidth="100"
    :gridRows="2"
    label="Choose Size"
    hint="Rows: 2, Columns: auto-calculated (3)"
    wire:model="selectedSize"
/>
```

### Form Validation Components

```blade
{{-- ReCaptcha Integration --}}
<x-vform-re-captcha />

{{-- Controller Validation --}}
use V360\FormComponents\Rules\ReCaptcha;

public function store(Request $request)
{
    $inputs = [
        'username' => ['required', 'min:3'],
        'email' => ['required', 'email'],
        'g-recaptcha-response' => ['required', new ReCaptcha],
    ];

    $messages = [
        'g-recaptcha-response.required' => 'Please confirm that you are not a robot!',
    ];

    $this->validate($request, $inputs, $messages);

    // Process the form...
}
```

### JavaScript Integration

Some components require JavaScript initialization. For Vite-based projects, follow these setup instructions:

#### 1. Install Required Dependencies

```bash
npm install nouislider filepond flatpickr choices.js virtual-select-plugin
```

#### 2. Import Styles in your `app.scss` or main CSS file

```scss
// Import third-party styles
@import "filepond/dist/filepond.min.css";
@import "flatpickr";
@import "nouislider/dist/nouislider.css";
@import "choices.js/src/styles/choices";
@import "virtual-select-plugin/dist/virtual-select.min.css";

// Import package styles (after publishing assets)
@import "../../sass/vform.scss";
```

#### 3. Import Scripts in your `app.js`

```javascript
// Import third-party libraries
import noUiSlider from "nouislider";
import * as FilePond from "filepond";
import flatpickr from "flatpickr";
import Choices from "choices.js";
import "virtual-select-plugin/dist/virtual-select.min.js";

// Import package components (after publishing assets)
import RangeSelector from "../js/vform/range-selector.js";
import SelectCheck from "../js/vform/select-check.js";

// Make libraries globally available
window.noUiSlider = noUiSlider;
window.RangeSelector = RangeSelector;
window.Choices = Choices;
window.SelectCheck = SelectCheck;

// Initialize components when DOM is ready
document.addEventListener("DOMContentLoaded", function () {
  // Initialize datepickers
  flatpickr(".datepicker", {
    enableTime: true,
    dateFormat: "Y-m-d H:i",
  });

  // Initialize FilePond for file inputs
  const fileInputs = document.querySelectorAll('input[type="file"]');
  fileInputs.forEach((input) => {
    FilePond.create(input);
  });

  // SelectChoice, SelectCheck, and RangeSelector are auto-initialized via their component JavaScript
  // No manual initialization required
});
```

#### Alternative: Direct HTML includes (for non-Vite projects)

```html
{{-- Include Required Scripts --}}
<script src="https://unpkg.com/filepond"></script>
<script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
<script src="https://cdn.jsdelivr.net/npm/nouislider@15.7.1/dist/nouislider.min.js"></script>
<link
  rel="stylesheet"
  href="https://cdn.jsdelivr.net/npm/nouislider@15.7.1/dist/nouislider.min.css"
/>

{{-- Include package assets (after publishing) --}}
<link rel="stylesheet" href="{{ asset('css/vform.css') }}" />
<script type="module" src="{{ asset('js/vform/range-selector.js') }}"></script>
```

### Styling

Components use Bulma classes by default, but you can customize the appearance:

```blade
<x-vform-text
    name="custom_input"
    :classes="'is-large is-rounded custom-class'"
    label="Custom Styled Input"
/>
```

For more detailed documentation about each component and its available options, please refer to the individual component documentation in the `docs/` directory.

### Documentation

In order to generate documentation for this package run the following command.

```bash
phpdoc -d . -t ./docs
```
