<?php

namespace V360\FormComponents\View\Components;

use Illuminate\View\Component;

/**
 * Summary of InputRichText
 * 
 * @dispatches Livewire:ChangeRichText:<name> with $delta
 * @dispatches CustomEvent:StartRichEdit:<name>
 * @dispatches CustomEvent:DoneRichEdit:<name>
 * @listens Livewire:LoadContent:<name> {content} where content is the $delta
 * @listens Livewire:DisposeContent:<name>
 */
class RichText extends Component
{

  public array $toolbarOptions = [];

  /**
   * Create a new component instance.
   * @param string $name Name of the field, will be used for the event communication
   * @param mixed $label Label of the field or false
   * @param bool $readOnly Read only or not
   * @param mixed $content Content of the field
   * @param bool $livewire Livewire or not
   * @param mixed $fonts Fonts to be used in the rich text editor
   */
  public function __construct(
    public string $name,
    public ?string $label = null,
    public bool $readOnly = false,
    public mixed $content = null,
    public bool $livewire = true,
    ?array $fonts = null,
  ) {
    $this->name = str($this->name)->replace(" ", "-")->__toString();
    $this->label = $label ?? str($this->name)->headline()->__toString();

    $fonts ??= config('vform.richtext.fonts', [
      'monospace',
      'sans-serif',
      'serif',
    ]);

    $this->toolbarOptions = [
      ['bold', 'italic', 'underline', 'strike'], // toggled buttons
      ['blockquote', 'code-block'],
      [
        ['header' => 1],
        ['header' => 2]
      ],
      [
        ['list' => 'ordered'],
        ['list' => 'bullet']
      ],
      [
        ['script' => 'sub'],
        ['script' => 'super']
      ],
      [
        ['indent' => '-1'],
        ['indent' => '+1']
      ],
      [
        ['size' => ['small', false, 'large', 'huge']]
      ],
      [
        ['header' => [1, 2, 3, 4, 5, 6, false]]
      ],
      [
        ['color' => []],
        ['background' => []]
      ],
      [
        ['font' => $fonts]
      ],
      [
        ['align' => []]
      ],
      ['link'],
      ['clean'] // remove formatting button
    ];
  }


  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.rich-text');
  }
}
