<?php

namespace V360\FormComponents\View\Components;

use Illuminate\View\Component;
use Illuminate\Contracts\View\View;

/**
 * InputSwitch Laravel Blade Component
 *
 * A modern toggle switch component that allows users to switch between two states (on/off).
 * Built with Laravel's component system and integrates seamlessly with Bulma CSS framework.
 * Features customizable labels, colors, hints, and full Livewire support for real-time updates.
 *
 * Key Features:
 * - Modern toggle switch UI with smooth animations
 * - Customizable on/off state labels
 * - Flexible layout options: inline or block layout
 * - Multiple color variants (primary, success, info, warning, danger)
 * - Form validation integration with error display
 * - Livewire real-time updates and wire:model binding
 * - Automatic label generation from field name
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 * - Accessibility support with proper focus states
 * - Hint text support for user guidance
 *
 * Component Structure:
 * - Main label: Displays above the switch (block layout) or beside it (inline layout)
 * - Toggle switch: Visual switch element with handle animation
 * - State labels: Optional "on" and "off" labels beside the switch
 * - Hint text: Optional help text below the switch
 * - Error display: Shows validation errors when present
 *
 * Layout Options:
 * - Inline Layout (inline=true): Label and switch on same line with gap spacing
 * - Block Layout (inline=false): Label above switch (traditional form layout)
 *
 * CSS Integration:
 * - Uses SCSS-based styling with Bulma variables
 * - Supports dark mode through CSS custom properties
 * - Smooth 0.3s transitions for all state changes
 * - Customizable colors through Bulma color system
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.1.0
 * @since 2024-06-30
 * @updated 2024-12-27
 *
 * @example Basic usage:
 * <x-vform-input-switch 
 *     name="enable_notifications" 
 *     label="Email Notifications" 
 *     color="success" 
 * />
 *
 * @example With state labels:
 * <x-vform-input-switch 
 *     name="feature_enabled" 
 *     label="Enable Feature"
 *     on="Enabled"
 *     off="Disabled"
 *     color="primary"
 *     hint="Toggle to enable or disable this feature" 
 * />
 *
 * @example With Livewire:
 * <x-vform-input-switch 
 *     name="user.is_active" 
 *     wire:model="user.is_active"
 *     label="User Status"
 *     on="Active"
 *     off="Inactive"
 *     color="success"
 * />
 *
 * @example Inline layout:
 * <x-vform-input-switch 
 *     name="agree_terms" 
 *     label="I agree to the terms"
 *     :inline="true"
 *     on="Yes"
 *     off="No"
 *     color="info"
 * />
 *
 * @example Block layout:
 * <x-vform-input-switch 
 *     name="newsletter" 
 *     label="Subscribe to Newsletter"
 *     :inline="false"
 *     color="primary"
 *     hint="Receive updates about new features"
 * />
 *
 * @example Without main label (inline):
 * <x-vform-input-switch 
 *     name="agree_terms" 
 *     :label="false"
 *     on="I Agree"
 *     off="Disagree"
 *     color="info"
 * />
 */
class InputSwitch extends Component
{

  /**
   * The unique HTML ID attribute for the input switch element.
   * Generated automatically based on the component name.
   * 
   * @var string
   */
  public string $id;

  /**
   * Create a new InputSwitch component instance.
   * 
   * @param string $name The name attribute for the input field. Used for form submission and wire:model binding.
   * @param string $color The color variant for the switch. Accepts Bulma color variants: 'primary', 'success', 'info', 'warning', 'danger'.
   * @param string|bool|null $label The label text for the component. Pass false to hide the label, null for auto-generated label from name.
   * @param bool $inline Whether to display the label and switch inline (true) or stacked vertically (false). Default is true.
   * @param string|null $wireModel The Livewire property to bind to. Defaults to the $name parameter if not specified.
   * @param string|null $on The label text to display for the "on" state. Optional visual indicator.
   * @param string|null $off The label text to display for the "off" state. Optional visual indicator.
   * @param string|null $hint Help text to display below the switch. Provides additional context to users.
   */
  public function __construct(
    public string $name,
    public string $color = 'primary',
    public string|bool|null $label = null,
    public bool $inline = true,
    public ?string $wireModel = null,
    public ?string $on = null,
    public ?string $off = null,
    public ?string $hint = null,
  ) {
    // Generate a unique ID based on the name
    $this->id = str($this->name)
      ->prepend('input_switch_')
      ->snake()
      ->replace('.', '_')
      ->toString();

    $this->wireModel ??= $name;
    $this->label = $this->label !== false
      ? ($this->label ?? str($name)->replace('.', '_')->headline()->toString())
      : null;
  }

  /**
   * Get the view that represents the InputSwitch component.
   * 
   * Returns the Blade template for rendering the toggle switch with all
   * configured properties including labels, colors, and state indicators.
   *
   * @return View The Blade view instance for the input-switch template
   */
  public function render()
  {
    return view('vform::components.input-switch');
  }
}
