<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * Datepicker Laravel Blade Component
 *
 * A datepicker component supporting date, time, and datetime selection.
 * Features customizable labels, hints, and validation integration. Supports setting
 * minimum and maximum date/time limits for enhanced user input control.
 *
 * Features:
 * - Supports date, time, and datetime selection modes
 * - Customizable labels and hints
 * - Minimum and maximum date/time limits
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-06-15
 *
 * @example
 * <!-- Datepicker for selecting a date -->
 * <x-vform-datepicker 
 *     name="event_date" 
 *     type="date" 
 *     :date-limits="['2024-01-01', '2024-12-31']" 
 *     label="Event Date" 
 *     hint="Select the date of the event." 
 * />
 *
 * <!-- Timepicker for selecting a time -->
 * <x-vform-datepicker 
 *     name="meeting_time" 
 *     type="time" 
 *     :time-limits="['09:00', '17:00']" 
 *     label="Meeting Time" 
 *     hint="Select a time between 9 AM and 5 PM." 
 * />
 *
 * <!-- Datetime picker for selecting both date and time -->
 * <x-vform-datepicker 
 *     name="appointment" 
 *     type="datetime" 
 *     label="Appointment" 
 *     hint="Select the date and time for your appointment." 
 * />
 *
 */
class Datepicker extends Component
{

  public string $name;

  /**
   * Type of picker - date | time | datetime 
   *
   * @var string
   */
  public string $type;
  public ?string $value = null;
  public string $id;
  public ?string $label = null;
  public ?string $hint = null;
  public ?string $minDate = null;
  public ?string $maxDate = null;
  public ?string $minTime = null;
  public ?string $maxTime = null;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param string $type - date | time | datetime
   * @param string|null $value
   * @param string|null $id
   * @param mixed $label
   * @param string|null $hint
   * @param array $dateLimits
   * @param array $timeLimits
   */
  public function __construct(
    string $name,
    string $type = 'date',
    ?string $value = null,
    ?string $id = null,
    mixed $label = null,
    ?string $hint = null,
    array $dateLimits = [],
    array $timeLimits = [],
  ) {
    $this->name = $name;
    $this->type = $type;
    $this->value = $value;
    $this->id = $id ? $id : Str::slug($name);
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));
    $this->hint = $hint;
    if ($dateLimits) {
      $this->minDate = $dateLimits[0];
      $this->maxDate = $dateLimits[1];
    }
    if ($timeLimits) {
      $this->minTime = $timeLimits[0];
      $this->maxTime = $timeLimits[1];
    }
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.datepicker');
  }
}
