<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * Amount Laravel Blade Component
 *
 * A currency amount input component with a prefixed currency icon.
 * Supports multiple currencies including rupee, dollar, euro, and pound.
 * Features customizable labels, hints, and placeholder text for enhanced user experience.
 *
 * Features:
 * - Currency icon prefix based on selected currency
 * - Customizable labels and hints
 * - Right-aligned input for better readability
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-06-20
 *
 * @example
 * <!-- Amount input with rupee currency -->
 * <x-vform-amount 
 *     name="total_amount" 
 *     currency="rupee" 
 *     label="Total Amount" 
 *     hint="Enter the total amount in rupees." 
 * />
 *
 * <!-- Amount input with dollar currency and custom placeholder -->
 * <x-vform-amount 
 *     name="price" 
 *     currency="dollar" 
 *     label="Price" 
 *     placeholder="Enter the price in dollars." 
 * />
 *
 */
class Amount extends Component
{
  public string $name;
  public string $icon;
  public ?string $value = null;
  public string $id;
  public ?string $label = null;
  public string $class;
  public ?string $hint = null;
  public ?string $placeholder = null;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param string $currency - rupee | dollar | euro | pound
   * @param string|null $value
   * @param string|null $id
   * @param mixed $label
   * @param string|null $placeholder
   * @param string|null $classes
   * @param string|null $hint
   */
  public function __construct(
    string $name,
    string $currency = "rupee",
    ?string $value = null,
    ?string $id = null,
    mixed $label = null,
    ?string $placeholder = null,
    ?string $classes = null,
    ?string $hint = null
  ) {
    $this->name = $name;
    $this->value = $value;
    $this->id = $id ? $id : str_replace('_', '-', $name);
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));
    $this->class = "input has-text-right $classes";
    if ($this->label === false) {
      $this->placeholder = Str::title(str_replace('_', ' ', $name));
    }
    if ($placeholder) {
      $this->placeholder = $placeholder;
    }
    $this->hint = $hint;

    switch ($currency) {
      case 'rupee':
        $this->icon = 'currency_rupee';
        break;
      case 'dollar':
        $this->icon = 'attach_money';
        break;
      case 'euro':
        $this->icon = 'euro';
        break;
      case 'pound':
        $this->icon = 'currency_pound';
        break;
      default:
        $this->icon = 'attach_money';
        break;
    }
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.amount');
  }
}
