<?php

namespace V360\FormComponents\View\Components;

use Closure;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Collection;
use Illuminate\View\Component;

/**
 * SelectGrid Laravel Blade Component
 *
 * A visual grid-based selection component that presents options as clickable cards
 * arranged in a customizable grid layout. Perfect for image-based selections, product
 * choices, or any scenario where visual representation enhances user experience.
 * Supports both single and multiple selection modes with responsive grid layouts.
 *
 * Features:
 * - Visual grid layout with customizable dimensions
 * - Single or multiple selection support
 * - Image/icon support for each option
 * - Responsive grid structure
 * - Automatic label generation from name
 * - Collection and array options support
 * - Customizable option width and grid rows
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * Grid Layout:
 * The component automatically calculates columns based on the total number of options
 * and the specified number of rows. Each grid item can display:
 * - Visual content (image, icon, or styled content)
 * - Label text
 * - Selection state indicators
 *
 * Option Structure:
 * Each option should contain:
 * - 'value': The form value when selected
 * - 'label': Display text for the option
 * - 'url': Optional image/icon URL for visual representation
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2025-12-23
 *
 * @example
 * <!-- Simple grid with images -->
 * <x-vform-select-grid 
 *     name="avatar" 
 *     :options="[
 *         ['value' => 'avatar1', 'label' => 'Avatar 1', 'url' => '/avatars/1.png'],
 *         ['value' => 'avatar2', 'label' => 'Avatar 2', 'url' => '/avatars/2.png']
 *     ]"
 *     :multiple="false"
 *     :optionWidth="120"
 *     :gridRows="2" />
 *
 * @example
 * <!-- Multiple selection grid -->
 * <x-vform-select-grid 
 *     name="features" 
 *     :options="$productFeatures" 
 *     :multiple="true"
 *     label="Select Features"
 *     hint="Choose multiple features for your product"
 *     :optionWidth="150"
 *     :gridRows="3" />
 *
 * @example
 * <!-- Grid with custom styling -->
 * <x-vform-select-grid 
 *     name="theme" 
 *     :options="$themes"
 *     label="Choose Theme"
 *     :multiple="false"
 *     :optionWidth="200"
 *     :gridRows="1"
 *     wire:model="selectedTheme" />
 */
class SelectGrid extends Component
{
  /**
   * Construct SelectGrid component
   * 
   * @param string $name Name of the select input
   * @param Collection|array $options Options for the select grid. Each item should have: [value, label, url]
   * @param string|null $id Id of the select input (auto-generated if null)
   * @param array|null $value Initial selected value(s)
   * @param string|null|false $label Label for the select input. False to hide, null for auto-generated
   * @param bool $multiple Whether multiple selection is allowed
   * @param int $optionWidth Width of each option in the grid (in pixels)
   * @param int $gridRows Number of rows in the grid
   * @param string|null $hint Optional hint text to display below the component
   */
  public function __construct(
    public string $name,
    public Collection|array $options,
    public ?string $id = null,
    public ?array $value = null,
    public string|null|false $label = null,
    public bool $multiple = false,
    public int $optionWidth = 150,
    public int $gridRows = 1,
    public ?string $hint = null
  ) {
    $this->id = $id ?: $this->generateId($this->name);
    $this->value ??= [];

    // Convert collection to array if needed
    if ($this->options instanceof Collection) {
      $this->options = $this->options->toArray();
    }
  }

  /**
   * Get the view / contents that represent the component.
   * 
   * @return View|Closure|string The component view instance
   */
  public function render(): View|Closure|string
  {
    return view('vform::components.select-grid');
  }

  /**
   * Generate HTML ID from component name
   *
   * Converts the component name to a valid HTML ID by:
   * - Converting to snake_case
   * - Replacing dots with underscores for nested names
   *
   * @param string $name Component name (e.g., 'user.profile' or 'userName')
   * @return string Generated HTML ID (e.g., 'user_profile' or 'user_name')
   */
  private function generateId(string $name): string
  {
    return str($this->name)
      ->snake()
      ->replace('.', '_')
      ->toString();
  }

  /**
   * Get the processed label for display
   * 
   * Processes the label based on the label property:
   * - If false: returns false (no label)
   * - If null: auto-generates from name (snake_case to Title Case)
   * - Otherwise: returns the provided label string
   * 
   * @return string|false The processed label string or false if no label should be shown
   */
  public function getProcessedLabel(): string|false
  {
    if ($this->label === false) {
      return false;
    }

    if ($this->label === null) {
      // Generate from name: convert snake_case/camelCase to Title Case
      return str($this->name)
        ->snake()
        ->replace('_', ' ')
        ->title()
        ->toString();
    }

    return $this->label;
  }

  /**
   * Check if a value is selected
   * 
   * Determines if the given option value is currently selected by checking
   * if it exists in the component's value array.
   * 
   * @param mixed $optionValue The option value to check for selection
   * @return bool True if the value is selected, false otherwise
   */
  public function isSelected($optionValue): bool
  {
    if (empty($this->value)) {
      return false;
    }

    return \in_array($optionValue, $this->value);
  }

  /**
   * Get the number of columns based on options count and rows
   * 
   * Calculates the number of columns per row by dividing the total
   * number of options by the specified number of grid rows and
   * rounding up to ensure all options fit in the grid.
   * 
   * @return int The number of columns per row in the grid
   */
  public function getColumnsPerRow(): int
  {
    $optionCount = \count($this->options);
    return (int) ceil($optionCount / $this->gridRows);
  }
}
