<?php

namespace V360\FormComponents\View\Components;

use Closure;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * SelectChoice Laravel Blade Component
 *
 * An enhanced select dropdown component powered by Choices.js for modern UI.
 * Always includes Choices.js functionality with search, remove buttons, and enhanced UX.
 * Supports both single/multiple selection, grouped options, and integrates
 * seamlessly with Laravel's form validation and Livewire.
 *
 * Features:
 * - Single or multiple selection support
 * - Grouped options (optgroups) support
 * - Search functionality with customizable threshold
 * - Remove buttons for multiple selections
 * - Maximum item count limiting for multiple selection
 * - Automatic label generation from name
 * - Collection and array options support
 * - Optional sorting of options
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Always enhanced with Choices.js library
 *
 * Livewire Integration:
 * 
 * **IMPORTANT**: When using multiple selection (multiple=true), DO NOT use wire:model.
 * The component handles Livewire communication through custom events.
 * 
 * Events Dispatched:
 * - `SelectChoiceUpdated:{{ $id }}` - Fired when selection changes
 *   Payload: { value: string|array|null }
 * 
 * Events Listened For:
 * - `SelectChoiceReset:{{ $id }}` - Clears all selected items
 * - `SelectChoiceRemoveValue:{{ $id }}` - Removes specific value(s)
 *   Payload: { value: string|array }
 *
 * @package V360\FormComponents\View\Components
 * @author Laravel Developer
 * @version 1.0.0
 * @since 2025-12-20
 *
 * @example
 * <!-- Simple single select with wire:model (ALLOWED) -->
 * <x-select-choice 
 *     name="country" 
 *     :options="['US' => 'United States', 'CA' => 'Canada']"
 *     wire:model="selectedCountry" />
 *
 * @example
 * <!-- Multiple select WITHOUT wire:model (REQUIRED) -->
 * <x-select-choice 
 *     name="skills" 
 *     :options="$skills" 
 *     :multiple="true" 
 *     :maxItemCount="3"
 *     placeholder="Select up to 3 skills" />
 * <!-- Listen for changes in Livewire component: -->
 * <!-- Livewire::on('SelectChoiceUpdated:skills', function($data) { $this->skills = $data['value']; }); -->
 *
 * @example
 * <!-- Grouped options with validation -->
 * <x-select-choice 
 *     name="category" 
 *     :options="['Fruits' => ['apple' => 'Apple'], 'Vegetables' => ['carrot' => 'Carrot']]"
 *     label="Product Category"
 *     hint="Choose from available categories"
 *     class="is-large" />
 *
 * @example
 * <!-- Livewire event control -->
 * <!-- Reset all selections: -->
 * <!-- $this->dispatch('SelectChoiceReset:category'); -->
 * <!-- Remove specific values: -->
 * <!-- $this->dispatch('SelectChoiceRemoveValue:skills', ['value' => ['php', 'javascript']]); -->
 */
class SelectChoice extends Component
{
  /**
   * The HTML id attribute (generated from name if not provided)
   */
  public string $id;

  /**
   * Additional CSS classes (processed)
   */
  public string $class = '';

  /**
   * Processed options array after normalization
   */
  public array $options = [];

  /**
   * Choices.js configuration as JSON string
   */
  public string $choicesConfig = '';

  /**
   * Create a new SelectChoice component instance
   *
   * @param string $name Required. The form input name
   * @param array|Collection $options Available options (key-value pairs or nested arrays for groups)
   * @param string|array|null $value Selected value(s) for single/multiple selection
   * @param bool $multiple Enable multiple selection (default: false for single selection)
   *                        **IMPORTANT**: When true, DO NOT use wire:model attribute
   * @param string|null $placeholder Placeholder text for empty selection
   * @param string|null $id HTML id attribute (auto-generated from name if null)
   * @param string|false|null $label Display label (auto-generated from name if null, false to hide)
   * @param string|array|null $classes Additional CSS classes (string or array)
   * @param string|null $hint Help text displayed below the select
   * @param int|null $maxItemCount Maximum number of items that can be selected (only for multiple selection)
   * @param bool $sort Whether to sort options alphabetically (default: false)
   *
   * @throws \InvalidArgumentException When name is empty
   */
  public function __construct(
    public string $name,
    array|Collection $options = [],
    public string|array|null $value = null,
    public bool $multiple = true,
    public string|null $placeholder = null,
    ?string $id = null,
    public string|false|null $label = null,
    string|array|null $classes = null,
    public string|null $hint = null,
    public int|null $maxItemCount = null,
    bool $sort = false,
  ) {
    if (empty(trim($this->name))) {
      throw new \InvalidArgumentException('Select name cannot be empty');
    }

    // Process options
    $this->options = $this->processOptions($options, $sort);
    $this->id = $id ?: $this->generateId($this->name);

    // Process CSS classes
    $this->class = $this->processClasses($classes);

    // Generate Choices.js configuration
    $this->choicesConfig = $this->generateChoicesConfig();
  }

  /**
   * Get the view / contents that represent the component
   *
   * @return View|Closure|string The component view instance
   */
  public function render(): View|Closure|string
  {
    return view('vform::components.select-choice');
  }

  /**
   * Generate Choices.js configuration
   *
   * Builds the JSON configuration object for Choices.js initialization,
   * including search settings, placeholder handling, and maxItemCount for multiple selection.
   *
   * @return string JSON configuration for Choices.js
   */
  private function generateChoicesConfig(): string
  {
    $config = [
      'removeItemButton' => $this->multiple,
      'searchEnabled' => \count($this->options) > 5,
      'searchPlaceholderValue' => 'Search...',
      'placeholder' => !empty($this->placeholder),
      'placeholderValue' => $this->placeholder ?? '',
      'itemSelectText' => '',
      'shouldSort' => false,
      'silent' => false,
    ];

    // Add maxItemCount for multiple selection if specified
    if ($this->multiple && $this->maxItemCount !== null) {
      $config['maxItemCount'] = $this->maxItemCount;
    }

    return json_encode($config);
  }

  /**
   * Process CSS classes from various input formats
   *
   * @param string|array|null $classes CSS classes input
   * @return string Processed CSS classes string
   */
  private function processClasses(string|array|null $classes): string
  {
    $baseClasses = 'custom-select';

    if ($classes === null) {
      return $baseClasses;
    }

    if (\is_array($classes)) {
      $classesString = implode(' ', array_filter($classes));
    } else {
      $classesString = $classes;
    }

    return trim($baseClasses . ' ' . $classesString);
  }

  /**
   * Process and normalize options from various input formats
   *
   * @param array|Collection $options Raw options input
   * @param bool $sort Whether to sort the options
   * @return array Processed options array
   */
  private function processOptions(array|Collection $options, bool $sort = false): array
  {
    $processedOptions = $options instanceof Collection ? $options->toArray() : $options;

    if ($sort) {
      asort($processedOptions, SORT_REGULAR);
    }
    return $processedOptions;
  }

  /**
   * Generate HTML ID from component name
   *
   * @param string $name Component name
   * @return string Generated HTML ID
   */
  private function generateId(string $name): string
  {
    return str($this->name)
      ->snake()
      ->replace('.', '_')
      ->toString();
  }

  /**
   * Get the processed label (auto-generated if null, false to hide)
   *
   * @return string|false|null The processed label string, false if hidden, null if not set
   */
  public function getProcessedLabel(): string|false|null
  {
    if ($this->label === false) {
      return false;
    }

    if ($this->label !== null) {
      return $this->label;
    }

    // Auto-generate label from name
    return Str::headline($this->name);
  }

  /**
   * Check if a value is selected
   *
   * @param mixed $optionValue The option value to check
   * @return bool Whether the value is selected
   */
  public function isSelected(mixed $optionValue): bool
  {
    if ($this->value === null) {
      return false;
    }

    if ($this->multiple && \is_array($this->value)) {
      return \in_array($optionValue, $this->value, false);
    }


    return (string) $this->value === (string) $optionValue;
  }
}
