<?php

namespace V360\FormComponents\View\Components;

use Closure;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * SelectCheck Laravel Blade Component
 *
 * An enhanced select dropdown component powered by VirtualSelect.js for modern UI.
 * Always includes VirtualSelect.js functionality with search, remove buttons, and enhanced UX.
 * Supports both single/multiple selection, grouped options, and integrates
 * seamlessly with Laravel's form validation and Livewire.
 *
 * Features:
 * - Single or multiple selection support
 * - Grouped options (optgroups) support
 * - Search functionality with customizable threshold
 * - Remove buttons for multiple selections
 * - Maximum item count limiting for multiple selection
 * - Automatic label generation from name
 * - Collection and array options support
 * - Optional sorting of options
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Always enhanced with VirtualSelect.js library
 *
 * Livewire Integration:
 * 
 * **IMPORTANT**: When using multiple selection (multiple=true), DO NOT use wire:model.
 * The component handles Livewire communication through custom events.
 * 
 * Events Dispatched:
 * - `SelectCheckUpdated:{{ $id }}` - Fired when selection changes (default behavior)
 *   Payload: { value: string|array|null }
 * - `SelectCheckUpdated` - Fired when selection changes (when dispatchGenericEvent=true)
 *   Payload: { value: string|array|null, id: string }
 * 
 * Events Listened For:
 * - `SelectCheckReset:{{ $id }}` - Clears all selected items
 * - `SelectCheckReset` - Clears all selected items (generic event)  
 * - `SelectCheckRemoveValue:{{ $id }}` - Removes specific value(s)
 *   Payload: { value: string|array }
 *
 * @package V360\FormComponents\View\Components
 * @author Laravel Developer
 * @version 1.0.0
 * @since 2025-12-22
 *
 * @example
 * <!-- Simple single select with wire:model (ALLOWED) -->
 * <x-vform-select-check 
 *     name="country" 
 *     :options="['US' => 'United States', 'CA' => 'Canada']"
 *     :multiple="false"
 *     wire:model="selectedCountry" />
 *
 * @example
 * <!-- Multiple select WITHOUT wire:model (REQUIRED) -->
 * <x-vform-select-check 
 *     name="skills" 
 *     :options="$skills" 
 *     :multiple="true" 
 *     :maxItemCount="3"
 *     placeholder="Select up to 3 skills" />
 * <!-- Listen for changes in Livewire component: -->
 * <!-- Livewire::on('SelectCheckUpdated:skills', function($data) { $this->skills = $data['value']; }); -->
 *
 * @example
 * <!-- Grouped options with validation -->
 * <x-vform-select-check 
 *     name="category" 
 *     :options="['Fruits' => ['apple' => 'Apple'], 'Vegetables' => ['carrot' => 'Carrot']]"
 *     label="Product Category"
 *     hint="Choose from available categories"
 *     class="is-large" />
 *
 * @example
 * <!-- Livewire event control -->
 * <!-- Reset all selections: -->
 * <!-- $this->dispatch('SelectCheckReset:category'); -->
 * <!-- Remove specific values: -->
 * <!-- $this->dispatch('SelectCheckRemoveValue:skills', ['value' => ['php', 'javascript']]); -->
 */
class SelectCheck extends Component
{
  /**
   * The HTML id attribute (generated from name if not provided)
   */
  public string $id;

  /**
   * Additional CSS classes (processed)
   */
  public string $class = '';

  /**
   * Processed options array after normalization
   */
  public array $options = [];

  /**
   * VirtualSelect.js configuration as JSON string
   */
  public string $virtualSelectConfig = '';

  /**
   * Create a new SelectCheck component instance
   *
   * @param string $name Required. The form input name
   * @param array|Collection $options Available options (key-value pairs or nested arrays for groups)
   * @param string|array|null $value Selected value(s) for single/multiple selection
   * @param bool $multiple Enable multiple selection (default: true for multiple selection)
   *                        **IMPORTANT**: When true, DO NOT use wire:model attribute
   * @param string|null $placeholder Placeholder text for empty selection
   * @param string|null $id HTML id attribute (auto-generated from name if null)
   * @param string|false|null $label Display label (auto-generated from name if null, false to hide)
   * @param string|array|null $classes Additional CSS classes (string or array)
   * @param string|null $hint Help text displayed below the select
   * @param int|null $maxItemCount Maximum number of items that can be selected (only for multiple selection)
   * @param bool $sort Whether to sort options alphabetically (default: false)
   * @param bool $dispatchGenericEvent Whether to dispatch generic event (SelectCheckUpdated) instead of specific (SelectCheckUpdated:id)
   *
   * @throws \InvalidArgumentException When name is empty
   * @throws \InvalidArgumentException When maxItemCount is not a positive integer
   * @throws \InvalidArgumentException When maxItemCount exceeds available options count
   */
  public function __construct(
    public string $name,
    array|Collection $options = [],
    public string|array|null $value = null,
    public bool $multiple = true,
    public string|null $placeholder = null,
    ?string $id = null,
    public string|false|null $label = null,
    string|array|null $classes = null,
    public string|null $hint = null,
    public int|null $maxItemCount = null,
    bool $sort = false,
    public bool $dispatchGenericEvent = false,
  ) {
    if (empty(trim($this->name))) {
      throw new \InvalidArgumentException('Select name cannot be empty');
    }

    // Process options
    $this->options = $this->processOptions($options, $sort);
    $this->id = $id ?: $this->generateId($this->name);

    // Process CSS classes
    $this->class = $this->processClasses($classes);

    // Generate VirtualSelect.js configuration
    $this->virtualSelectConfig = $this->generateVirtualSelectConfig();
  }

  /**
   * Get the view / contents that represent the component
   *
   * @return View|Closure|string The component view instance
   */
  public function render(): View|Closure|string
  {
    return view('vform::components.select-check');
  }

  /**
   * Generate VirtualSelect.js configuration
   *
   * Builds the JSON configuration object for VirtualSelect.js initialization,
   * including search settings, placeholder handling, and maxValues for multiple selection.
   *
   * @return string JSON configuration for VirtualSelect.js
   *
   * @throws \InvalidArgumentException When maxItemCount validation fails
   */
  private function generateVirtualSelectConfig(): string
  {
    $config = [
      'multiple' => $this->multiple,
      'search' => true,
      'showSelectedOptionsFirst' => true,
      'placeholder' => $this->placeholder ?? 'Select an option',
    ];

    // Add maxItemCount for multiple selection if specified
    if ($this->multiple && $this->maxItemCount !== null) {
      if ($this->maxItemCount <= 0) {
        throw new \InvalidArgumentException('maxItemCount must be a positive integer');
      }
      if ($this->maxItemCount > \count($this->options)) {
        throw new \InvalidArgumentException('maxItemCount cannot exceed the total number of available options');
      }
      $config['maxValues'] = $this->maxItemCount;
    }

    return json_encode($config);
  }

  /**
   * Process CSS classes from various input formats
   *
   * Combines the base CSS classes with user-provided classes.
   * Handles both string and array inputs for flexibility.
   *
   * @param string|array|null $classes CSS classes input (string, array, or null)
   * @return string Processed CSS classes string with base classes included
   */
  private function processClasses(string|array|null $classes): string
  {
    $baseClasses = 'custom-select';

    if ($classes === null) {
      return $baseClasses;
    }

    if (\is_array($classes)) {
      $classesString = implode(' ', array_filter($classes));
    } else {
      $classesString = $classes;
    }

    return trim($baseClasses . ' ' . $classesString);
  }

  /**
   * Process and normalize options from various input formats
   *
   * Handles both Laravel Collections and plain arrays.
   * Optionally sorts the options alphabetically for consistent display.
   *
   * @param array|Collection $options Raw options input (array or Laravel Collection)
   * @param bool $sort Whether to sort the options alphabetically (default: false)
   * @return array Processed options array ready for use in the component
   */
  private function processOptions(array|Collection $options, bool $sort = false): array
  {
    $processedOptions = $options instanceof Collection ? $options->toArray() : $options;
    if ($sort) {
      asort($processedOptions, SORT_REGULAR);
    }
    return $processedOptions;
  }

  /**
   * Generate HTML ID from component name
   *
   * Converts the component name to a valid HTML ID by:
   * - Converting to snake_case
   * - Replacing dots with underscores for nested names
   *
   * @param string $name Component name (e.g., 'user.profile' or 'userName')
   * @return string Generated HTML ID (e.g., 'user_profile' or 'user_name')
   */
  private function generateId(string $name): string
  {
    return str($this->name)
      ->snake()
      ->replace('.', '_')
      ->toString();
  }

  /**
   * Get the processed label for the component
   *
   * Handles three states:
   * - false: Explicitly hide the label
   * - string: Use the provided label text
   * - null: Auto-generate a human-readable label from the component name
   *
   * @return string|false|null The processed label string, false if hidden, null if not set
   */
  public function getProcessedLabel(): string|false|null
  {
    if ($this->label === false) {
      return false;
    }

    if ($this->label !== null) {
      return $this->label;
    }

    // Auto-generate label from name
    return Str::headline($this->name);
  }

  /**
   * Check if a specific option value is currently selected
   *
   * Handles both single and multiple selection modes:
   * - Single selection: Compares the value directly
   * - Multiple selection: Checks if the value exists in the selected values array
   *
   * @param mixed $optionValue The option value to check against selected value(s)
   * @return bool True if the option value is selected, false otherwise
   */
  public function isSelected(mixed $optionValue): bool
  {
    if ($this->value === null) {
      return false;
    }

    if ($this->multiple && \is_array($this->value)) {
      return \in_array($optionValue, $this->value, false);
    }

    return (string) $this->value === (string) $optionValue;
  }
}
