<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * Checkbox Laravel Blade Component
 *
 * A checkbox input component that supports single and multiple selections.
 * Features customizable labels, hints, and margin options for enhanced user experience.
 *
 * Features:
 * - Single and multiple checkbox support
 * - Customizable labels and hints
 * - Margin control for layout flexibility
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-06-30
 *
 * @example
 * <!-- Single checkbox -->
 * <x-vform-checkbox 
 *     name="agree_terms" 
 *     value="1" 
 *     label="I agree to the terms and conditions." 
 *     hint="You must agree before submitting." 
 * />
 *
 * <!-- Multiple checkboxes -->
 * <x-vform-checkbox 
 *     name="interests" 
 *     :options="['sports' => 'Sports', 'music' => 'Music', 'travel' => 'Travel']" 
 *     :value="['music', 'travel']" 
 *     label="Select your interests:" 
 *     hint="You can select multiple options." 
 * />
 *
 */
class Checkbox extends Component
{

  public string $name;
  public $options;
  public $value;
  public ?string $label = null;
  public $margin;
  public ?string $hint = null;

  public ?string $id = null;
  public string $class;
  public bool $multiple;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param array $options
   * @param mixed $value
   * @param mixed $label
   * @param string|null $classes
   * @param bool $margin
   * @param string|null $hint
   */
  public function __construct(
    string $name,
    $options = [],
    $value = null,
    $label = null,
    ?string $classes = null,
    bool $margin = true,
    ?string $hint = null
  ) {
    $this->name = $name;
    $this->class = "is-checkradio " . $classes;
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));

    if ($options && count($options) > 0) {
      $this->multiple = true;
      $this->name = $this->name . "[]";
      //prepare options
      $checkOptions = [];
      foreach ($options as $val => $label) {
        //is_checked
        $checked = is_array($value) && in_array($val, $value);
        $checkOptions[] = [
          'id' => Str::uuid(),
          'value' => $val,
          'label' => $label,
          'checked' => $checked,
        ];
      }
      $this->options = $checkOptions;
      $this->margin = '';
    } else {
      $this->multiple = false;
      $this->value = $value;
      $this->id = Str::uuid();
      $this->options = [];
      $this->margin = $margin ? 'pt-5 mt-2' : '';
    }

    $this->hint = $hint;
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.checkbox');
  }
}
