<?php

namespace V360\FormComponents\View\Components;

use Closure;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Collection;
use Illuminate\View\Component;

/**
 * SelectGrid Laravel Blade Component
 *
 * A visual grid-based selection component that presents options as clickable cards
 * arranged in a customizable grid layout. Perfect for image-based selections, product
 * choices, or any scenario where visual representation enhances user experience.
 * Supports both single and multiple selection modes with responsive grid layouts.
 *
 * Features:
 * - Visual grid layout with customizable dimensions
 * - Single or multiple selection support
 * - Image/icon support for each option
 * - Responsive grid structure
 * - Automatic label generation from name
 * - Collection and array options support
 * - Customizable option width and grid rows
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * Grid Layout:
 * The component supports flexible grid layouts by specifying either rows or columns:
 * - Specify gridRows: component calculates columns automatically
 * - Specify gridCols: component calculates rows automatically
 * - If neither specified, defaults to 1 row with auto-calculated columns
 * Each grid item can display:
 * - Visual content (image, icon, or styled content)
 * - Label text
 * - Selection state indicators
 *
 * Option Structure:
 * Each option should contain:
 * - 'value': The form value when selected
 * - 'label': Display text for the option
 * - 'url': Optional image/icon URL for visual representation
 * - 'color': Optional CSS color value for color swatch display (either url or color should be provided)
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2025-12-23
 *
 * @example
 * <!-- Simple grid with colors using columns -->
 * <x-vform-select-grid 
 *     name="theme_color" 
 *     :options="[
 *         ['value' => 'blue', 'label' => 'Ocean Blue', 'color' => '#3498db'],
 *         ['value' => 'green', 'label' => 'Forest Green', 'color' => '#2ecc71'],
 *         ['value' => 'red', 'label' => 'Crimson Red', 'color' => '#e74c3c']
 *     ]"
 *     :multiple="false"
 *     :optionWidth="120"
 *     :gridCols="3" />
 *
 * @example
 * <!-- Multiple selection grid with mixed content -->
 * <x-vform-select-grid 
 *     name="features" 
 *     :options="[
 *         ['value' => 'feature1', 'label' => 'Premium', 'color' => '#f39c12'],
 *         ['value' => 'feature2', 'label' => 'Logo', 'url' => '/icons/logo.png'],
 *         ['value' => 'feature3', 'label' => 'Custom', 'color' => '#9b59b6']
 *     ]" 
 *     :multiple="true"
 *     label="Select Features"
 *     hint="Choose multiple features for your product"
 *     :optionWidth="150"
 *     :gridRows="3" />
 *
 * @example
 * <!-- Grid with images using rows -->
 * <x-vform-select-grid 
 *     name="avatar" 
 *     :options="[
 *         ['value' => 'avatar1', 'label' => 'Avatar 1', 'url' => '/avatars/1.png'],
 *         ['value' => 'avatar2', 'label' => 'Avatar 2', 'url' => '/avatars/2.png'],
 *         ['value' => 'avatar3', 'label' => 'Avatar 3', 'url' => '/avatars/3.png'],
 *         ['value' => 'avatar4', 'label' => 'Avatar 4', 'url' => '/avatars/4.png']
 *     ]"
 *     label="Choose Avatar"
 *     :multiple="false"
 *     :optionWidth="200"
 *     :gridRows="2"
 *     wire:model="selectedAvatar" />
 */
class SelectGrid extends Component
{
  /**
   * Construct SelectGrid component
   * 
   * @param string $name Name of the select input
   * @param Collection|array $options Options for the select grid. Each item should have: [value, label, url?, color?]
   * @param string|null $id Id of the select input (auto-generated if null)
   * @param array|null $value Initial selected value(s)
   * @param string|null|false $label Label for the select input. False to hide, null for auto-generated
   * @param bool $multiple Whether multiple selection is allowed
   * @param int $optionWidth Width of each option in the grid (in pixels)
   * @param int|null $gridRows Number of rows in the grid (computed if gridCols is provided)
   * @param int|null $gridCols Number of columns in the grid (computed if gridRows is provided)
   * @param string|null $hint Optional hint text to display below the component
   */
  public function __construct(
    public string $name,
    public Collection|array $options,
    public ?string $id = null,
    public ?array $value = null,
    public string|null|false $label = null,
    public bool $multiple = false,
    public int $optionWidth = 150,
    public ?int $gridRows = null,
    public ?int $gridCols = null,
    public ?string $hint = null
  ) {
    $this->id = $id ?: $this->generateId($this->name);
    $this->value ??= [];

    // Convert collection to array if needed
    if ($this->options instanceof Collection) {
      $this->options = $this->options->toArray();
    }

    // Set default values for rows/cols if neither is specified
    if ($this->gridRows === null && $this->gridCols === null) {
      $this->gridRows = 1;
    }
  }

  /**
   * Get the view / contents that represent the component.
   * 
   * @return View|Closure|string The component view instance
   */
  public function render(): View|Closure|string
  {
    return view('vform::components.select-grid');
  }

  /**
   * Generate HTML ID from component name
   *
   * Converts the component name to a valid HTML ID by:
   * - Converting to snake_case
   * - Replacing dots with underscores for nested names
   *
   * @param string $name Component name (e.g., 'user.profile' or 'userName')
   * @return string Generated HTML ID (e.g., 'user_profile' or 'user_name')
   */
  private function generateId(string $name): string
  {
    return str($this->name)
      ->snake()
      ->replace('.', '_')
      ->toString();
  }

  /**
   * Get the processed label for display
   * 
   * Processes the label based on the label property:
   * - If false: returns false (no label)
   * - If null: auto-generates from name (snake_case to Title Case)
   * - Otherwise: returns the provided label string
   * 
   * @return string|false The processed label string or false if no label should be shown
   */
  public function getProcessedLabel(): string|false
  {
    if ($this->label === false) {
      return false;
    }

    if ($this->label === null) {
      // Generate from name: convert snake_case/camelCase to Title Case
      return str($this->name)
        ->snake()
        ->replace('_', ' ')
        ->title()
        ->toString();
    }

    return $this->label;
  }

  /**
   * Check if a value is selected
   * 
   * Determines if the given option value is currently selected by checking
   * if it exists in the component's value array.
   * 
   * @param mixed $optionValue The option value to check for selection
   * @return bool True if the value is selected, false otherwise
   */
  public function isSelected($optionValue): bool
  {
    if (empty($this->value)) {
      return false;
    }

    return \in_array($optionValue, $this->value);
  }

  /**
   * Get the number of columns for the grid
   * 
   * Returns the number of columns based on:
   * - If gridCols is specified, returns that value
   * - If gridRows is specified, calculates columns from total options divided by rows
   * - Ensures at least 1 column is returned
   * 
   * @return int The number of columns in the grid
   */
  public function getColumnsPerRow(): int
  {
    if ($this->gridCols !== null) {
      return $this->gridCols;
    }

    $optionCount = \count($this->options);
    $rows = $this->gridRows ?? 1;
    return (int) ceil($optionCount / $rows);
  }

  /**
   * Get the number of rows for the grid
   * 
   * Returns the number of rows based on:
   * - If gridRows is specified, returns that value
   * - If gridCols is specified, calculates rows from total options divided by columns
   * - Ensures at least 1 row is returned
   * 
   * @return int The number of rows in the grid
   */
  public function getRowsPerGrid(): int
  {
    if ($this->gridRows !== null) {
      return $this->gridRows;
    }

    $optionCount = \count($this->options);
    $cols = $this->gridCols ?? 1;
    return (int) ceil($optionCount / $cols);
  }
}
