<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * Radio Laravel Blade Component
 *
 * A radio button group component with customizable options.
 * Features customizable labels, hints, and styling for enhanced user experience.
 *
 * Features:
 * - Customizable radio options
 * - Customizable labels and hints
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-07-20
 *
 * @example
 * <!-- Radio button group -->
 * <x-vform-radio 
 *     name="gender"
 *    :options="['male' => 'Male', 'female' => 'Female']" 
 *     label="Gender"
 *    hint="Select your gender."
 * />
 * */

class Radio extends Component
{
  public string $name;
  public ?string $value = null;
  public ?string $id = null;
  public ?string $label = null;
  public $options;
  public string $classes;
  public ?string $hint = null;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param string|null $value
   * @param mixed $options
   * @param mixed $label
   * @param string $classes
   * @param string|null $hint
   */
  public function __construct(
    string $name,
    ?string $value = null,
    mixed $options = [],
    mixed $label = null,
    string $classes = '',
    ?string $hint = null
  ) {
    $this->name = $name;
    $this->value = $value;
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));

    //prepare options
    $radioOptions = [];
    foreach ($options as $value => $label) {
      $radioOptions[] = [
        'id' => Str::uuid(),
        'value' => $value,
        'label' => $label,
      ];
    }
    $this->options = $radioOptions;
    $this->classes = $classes;
    $this->hint = $hint;
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.radio');
  }
}
