<?php

namespace V360\FormComponents\View\Components;

use Illuminate\View\Component;
use Illuminate\Support\Str;

/**
 * FilePond Laravel Blade Component
 *
 * A file upload component using FilePond for enhanced user experience.
 * Supports multiple file uploads, drag-and-drop functionality, and real-time progress indicators.
 * Features customizable labels and options for flexibility.
 *
 * Features:
 * - Multiple file upload support
 * - Drag-and-drop functionality
 * - Real-time upload progress indicators
 * - Customizable labels
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * Livewire Integration:
 * 
 * **IMPORTANT**: The component handles Livewire communication through custom events.
 * 
 * Events Dispatched:
 * - `CancelFile:{{ $id }}` - Fired when file upload is cancelled
 *   Payload: none
 * 
 * Events Listened For:
 * - `ResetFile:{{ $id }}` - Resets the file input to empty state
 *   Payload: none
 * - `SetTempFile:{{ $id }}` - Sets a temporary file in the input
 *   Payload: { fileName: string, url: string }
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-07-15
 *
 * @example
 * <!-- Single file upload -->
 * <x-vform-file-pond 
 *     name="profile_picture" 
 *     label="Upload Profile Picture" 
 * />
 *
 * <!-- Multiple file upload with custom options -->
 * <x-vform-file-pond 
 *     name="documents" 
 *     :options="['maxFiles' => 5, 'acceptedFileTypes' => ['image/*', 'application/pdf']]" 
 *     label="Upload Documents" 
 * />
 *
 */
class FilePond extends Component
{

  /**
   * Constructor for the component.
   * 
   * This component is used to create a file upload field using FilePond.
   * It allows for easy file uploads with various options and features.
   * 
   * @param string $name  The name of the file field. It should match the name of the file field in the form.
   * @param string $id The unique id of the file field.
   * @param mixed $label The label for the file field. If not provided, it will be generated from the name.
   * @param array $options The options for the file field.
   */
  public function __construct(
    public string $name,
    public string $id,
    public mixed $label = null,
    public array $options = [],
  ) {
    if ($this->label !== false) {
      $this->label = $this->label ?? Str::headline($this->name);
    }
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.file-pond');
  }
}
