<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;


/**
 * Email Laravel Blade Component
 *
 * An email input component with optional domain restriction.
 * Features customizable labels, hints, and placeholder text for enhanced user experience.
 *
 * Features:
 * - Optional domain restriction for email addresses
 * - Customizable labels and hints
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-07-10
 *
 * @example
 * <!-- Basic email input -->
 * <x-vform-email 
 *     name="user_email" 
 *     label="Email Address" 
 *     hint="Enter your email address." 
 * />
 *
 * <!-- Email input with domain restriction -->
 * <x-vform-email 
 *     name="company_email" 
 *     domain="example.com" 
 *     label="Company Email" 
 *     hint="Enter your company email address." 
 * />
 *
 */
class Email extends Component
{

  public string $name;
  public ?string $value = null;
  public string $id;
  public ?string $label = null;
  public string $icon;
  public string $class;
  public ?string $hint = null;
  public ?string $placeholder = null;
  public ?string $domain = null;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param string|null $value
   * @param string|null $id
   * @param mixed $label
   * @param bool $icon
   * @param string|null $placeholder
   * @param string|null $classes
   * @param string|null $hint
   * @param string|null $domain
   */
  public function __construct(
    string $name,
    ?string $value = null,
    ?string $id = null,
    mixed $label = null,
    bool $icon = true,
    ?string $placeholder = null,
    ?string $classes = null,
    ?string $hint = null,
    ?string $domain = null
  ) {
    $this->name = $name;
    $this->value = $value;

    $this->id = $id ? $id : str_replace('_', '-', $name);
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));
    $this->icon = $icon;
    $this->class = "input $classes";
    if ($this->label === false) {
      $this->placeholder = Str::title(str_replace('_', ' ', $name));
    }
    if ($placeholder) {
      $this->placeholder = $placeholder;
    }
    $this->hint = $hint;
    $this->domain = $domain;
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.email');
  }
}
