<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * Contactno Laravel Blade Component
 *
 * A contact number input component with a prefixed icon based on the type of contact number.
 * Supports various types including Support, Helpline, Work, Office, Mobile, Personal, and Home.
 * Features customizable labels, hints, and placeholder text for enhanced user experience.
 *
 * Features:
 * - Icon prefix based on contact number type
 * - Customizable labels and hints
 * - Input pattern validation for phone numbers
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-07-05
 *
 * @example
 * <!-- Contact number input for Support -->
 * <x-vform-contactno 
 *     name="support_contact" 
 *     type="Support" 
 *     label="Support Contact" 
 *     hint="Enter the support contact number." 
 * />
 *
 * <!-- Contact number input for Mobile -->
 * <x-vform-contactno 
 *     name="mobile_number" 
 *     type="Mobile" 
 *     label="Mobile Number" 
 *     hint="Enter your mobile number." 
 * />
 *
 */
class Contactno extends Component
{
  public string $name;
  public ?string $value = null;
  public string $type;
  public string $id;
  public ?string $label = null;
  public string $class;
  public ?string $hint = null;
  public string $icon;
  public string $pattern;
  public ?string $placeholder = null;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param string|null $value
   * @param string|null $type - Support | Helpline | Work | Office | Mobile | Personal | Home
   * @param string|null $id
   * @param mixed $label
   * @param string|null $classes
   * @param string|null $hint
   */
  public function __construct(
    string $name,
    ?string $value = null,
    ?string $type = null,
    ?string $id = null,
    mixed $label = null,
    ?string $classes = null,
    ?string $hint = null
  ) {
    $this->name = $name;
    $this->value = $value;
    $this->type = Str::title($type);
    $this->id = $id ? $id : str_replace('_', '-', $name);
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));
    $this->hint = $hint;
    $this->class = "input is-fullwidth $classes";

    $this->pattern = "(\+[0-9]{1,3})*([0-9]){6,15}";
    $this->placeholder = $this->type . " Number";

    switch ($this->type) {
      case 'Support':
        $this->icon = 'contact_support';
        break;
      case 'Helpline':
        $this->icon = 'contact_phone';
        break;
      case 'Work':
      case 'Office':
        $this->icon = 'deskphone';
        break;
      case 'Mobile':
        $this->icon = 'smartphone';
        $this->pattern = "(\+[0-9]{1,3})*[0-9]{8,12}";
        $this->placeholder = "XXXXXXXXXX";
        break;
      case 'Personal':
        $this->icon = 'call';
        break;
      case 'Home':
        $this->icon = 'contact_phone';
        break;
      default:
        $this->type = "Contact";
        $this->icon = 'call';
        break;
    }
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.contactno');
  }
}
