<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * Textarea Laravel Blade Component
 *
 * A textarea input component with customizable labels, hints, and placeholder text for enhanced user experience.
 *
 * Features:
 * - Customizable labels and hints
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-07-12
 *
 * @example
 * <!-- Basic textarea input -->
 * <x-vform-textarea 
 *     name="user_bio" 
 *     label="Biography" 
 *     hint="Tell us about yourself." 
 *     placeholder="Enter your biography here..."
 * />
 *
 */
class Textarea extends Component
{

  public string $name;
  public ?string $value = null;
  public string $id;
  public ?string $label = null;
  public ?string $placeholder = null;
  public string $class;
  public ?string $hint = null;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param string|null $value
   * @param string|null $id
   * @param mixed $label
   * @param string|null $classes
   * @param string|null $placeholder
   * @param string|null $hint
   */
  public function __construct(
    string $name,
    ?string $value = null,
    ?string $id = null,
    mixed $label = null,
    ?string $classes = null,
    ?string $placeholder = null,
    ?string $hint = null
  ) {
    $this->name = $name;
    $this->value = $value;
    $this->id = $id ? $id : str_replace('_', '-', $name);
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));
    $this->class = "textarea " . $classes;
    if ($this->label === false) {
      $this->placeholder = Str::title(str_replace('_', ' ', $name));
    }
    if ($placeholder) {
      $this->placeholder = $placeholder;
    }
    $this->hint = $hint;
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.textarea');
  }
}
