<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Str;
use Illuminate\View\Component;
use Illuminate\Support\Collection;

/**
 * Tags Laravel Blade Component
 *
 * A tags input component allowing users to add and remove tags dynamically.
 * Features customizable labels, hints, and placeholder text for enhanced user experience.
 *
 * Features:
 * - Dynamic tag addition and removal
 * - Customizable labels and hints
 * - Form validation integration
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-08-01
 *
 * @example
 * <!-- Basic tags input -->
 * <x-vform-tags 
 *     name="tags" 
 *     label="Tags" 
 *     :options="['Laravel', 'PHP', 'JavaScript']" 
 *     placeholder="Add a tag" 
 *     hint="Enter tags related to your post." 
 * />
 *
 */
class Tags extends Component
{
  /**
   * Create a new component instance.

   * @param string $name
   * @param mixed $label
   * @param string|null $id
   * @param mixed $values
   * @param array $options
   * @param boolean $keyValue
   * @param string|null $placeholder
   * @param string|null $hint
   */
  public function __construct(
    public string $name,
    public mixed $label = null,
    public ?string $id = null,
    public mixed $values = [],
    public array|Collection $options = [],
    public bool $keyValue = true,
    public ?string $placeholder = null,
    public ?string $hint = null,
  ) {
    $this->id = $id ?? Str::uuid();
    $this->label = $label !== false ? ($label ?? Str::headline($name)) : false;
    if (is_string($values)) {
      $this->values = json_decode($values);
    } else if (!is_array($values)) {
      $this->values = $this->values->toArray();
    }
    if (is_array($options) && !$options) {
      $this->keyValue = false;
    } else if ($options && !is_array($options) && $options->isEmpty()) {
      $this->keyValue = false;
    }
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.tags');
  }
}
