<?php

namespace V360\FormComponents\View\Components;

use Illuminate\View\Component;

/**
 * RichText Laravel Blade Component
 *
 * A rich text editor component with customizable toolbar options.
 * Features customizable labels and read-only mode for enhanced user experience.
 *
 * Features:
 * - Customizable toolbar options
 * - Read-only mode
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * Livewire Integration:
 * 
 * **IMPORTANT**: The component handles Livewire communication through custom events.
 * 
 * Events Dispatched:
 * - `Livewire:ChangeRichText:{{ $name }}` - Fired when rich text content changes
 *   Payload: { delta: object }
 * - `CustomEvent:StartRichEdit:{{ $name }}` - Fired when editing starts
 *   Payload: none
 * - `CustomEvent:DoneRichEdit:{{ $name }}` - Fired when editing is completed
 *   Payload: none
 * 
 * Events Listened For:
 * - `Livewire:LoadContent:{{ $name }}` - Load content into the editor
 *   Payload: { content: object } (where content is the delta)
 * - `Livewire:DisposeContent:{{ $name }}` - Dispose/clear the editor content
 *   Payload: none
 * <!-- Basic rich text editor -->
 * <x-vform-rich-text 
 *     name="article_content" 
 *     label="Article Content" 
 * />
 *
 * <!-- Read-only rich text editor -->
 * <x-vform-rich-text 
 *     name="article_preview" 
 *     label="Article Preview" 
 *     :read-only="true" 
 *     :content="$articleContent" 
 * />
 *
 */
class RichText extends Component
{

  public array $toolbarOptions = [];

  /**
   * Create a new component instance.
   * @param string $name Name of the field, will be used for the event communication
   * @param mixed $label Label of the field or false
   * @param bool $readOnly Read only or not
   * @param mixed $content Content of the field
   * @param bool $livewire Livewire or not
   * @param mixed $fonts Fonts to be used in the rich text editor
   */
  public function __construct(
    public string $name,
    public ?string $label = null,
    public bool $readOnly = false,
    public mixed $content = null,
    public bool $livewire = true,
    ?array $fonts = null,
  ) {
    $this->name = str($this->name)->replace(" ", "-")->__toString();
    $this->label = $label ?? str($this->name)->headline()->__toString();

    $fonts ??= config('vform.richtext.fonts', [
      'monospace',
      'sans-serif',
      'serif',
    ]);

    $this->toolbarOptions = [
      ['bold', 'italic', 'underline', 'strike'], // toggled buttons
      ['blockquote', 'code-block'],
      [
        ['header' => 1],
        ['header' => 2]
      ],
      [
        ['list' => 'ordered'],
        ['list' => 'bullet']
      ],
      [
        ['script' => 'sub'],
        ['script' => 'super']
      ],
      [
        ['indent' => '-1'],
        ['indent' => '+1']
      ],
      [
        ['size' => ['small', false, 'large', 'huge']]
      ],
      [
        ['header' => [1, 2, 3, 4, 5, 6, false]]
      ],
      [
        ['color' => []],
        ['background' => []]
      ],
      [
        ['font' => $fonts]
      ],
      [
        ['align' => []]
      ],
      ['link'],
      ['clean'] // remove formatting button
    ];
  }


  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.rich-text');
  }
}
