<?php

namespace V360\FormComponents\View\Components;

use Illuminate\View\Component;
use Illuminate\Support\Str;

/**
 * File Laravel Blade Component
 *
 * A file input component that supports different document types such as PDF, Excel, Image, and Zip.
 * Features customizable labels, placeholders, and hints for enhanced user experience.
 *
 * Features:
 * - Supports multiple document types with appropriate file acceptance
 * - Customizable labels and placeholders
 * - Option to display the file input in a box style
 * - Form validation integration
 * - Livewire real-time updates
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-06-25
 *
 * @example
 * <!-- File input for PDF documents -->
 * <x-vform-file 
 *     name="resume" 
 *     type="pdf" 
 *     label="Upload your resume" 
 *     hint="Accepted format: PDF" 
 * />
 *
 * <!-- File input for images with box style -->
 * <x-vform-file 
 *     name="profile_picture" 
 *     type="image" 
 *     label="Profile Picture" 
 *     placeholder="Choose an image" 
 *     :box="true" 
 *     hint="Accepted formats: JPG, PNG" 
 * />
 *
 */
class File extends Component
{
  public string $name;
  public ?string $label = null;
  public ?string $placeholder = null;
  public string $id;

  /**
   * Type of document - pdf | excel | image
   *
   * @var string|null
   */
  public ?string $type = null;
  public string $accept;
  public bool $box;
  public ?string $filename = null;
  public ?string $hint = null;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param mixed $label
   * @param string $placeholder
   * @param string|null $id
   * @param string|null $type - pdf | excel | image | zip
   * @param bool $box - whether to show the file in a box or not
   * @param string|null $filename - name of the file to be displayed
   * @param string|null $hint - additional information about the file input
   */
  public function __construct(
    string $name,
    mixed $label = null,
    string $placeholder = 'Choose a file',
    ?string $id = null,
    ?string $type = null,
    bool $box = false,
    ?string $filename = null,
    ?string $hint = null
  ) {
    $this->name = $name;
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));
    $this->placeholder = $placeholder;
    $this->box = $box;
    $this->id = $id ? $id : str_replace('_', '-', $name);
    switch ($type) {
      case 'pdf':
        $this->accept = '.pdf';
        break;
      case 'excel':
        $this->accept = '.xls,.xlsx';
        break;
      case 'image':
        $this->accept = 'image/*';
        break;
      case 'zip':
        $this->accept = 'application/zip, application/x-zip-compressed, multipart/x-zip';
        break;
      default:
        $this->accept = '*';
        break;
    }
    $this->filename = $filename;
    $this->hint = $hint;
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.file');
  }
}
