/**
 * SelectChoice - A class for managing Choices.js instances with Livewire integration
 *
 * Event Dispatching:
 * - When dispatchGenericEvent is false (default): Dispatches "SelectChoiceUpdated:elementId" with {value}
 * - When dispatchGenericEvent is true: Dispatches "SelectChoiceUpdated" with {value, id}
 */
export default class SelectChoice {
  constructor(elementId, choicesConfig = {}, selectChoiceConfig = {}) {
    this.elementId = elementId;
    this.choicesConfig = choicesConfig;
    this.selectChoiceConfig = selectChoiceConfig;
    this.isMultiple = selectChoiceConfig.multiple ?? false;
    this.dispatchGenericEvent =
      selectChoiceConfig.dispatchGenericEvent ?? false;
    this.debug = selectChoiceConfig.debug ?? false;
    this.isInitializing = true;
    this.choicesElement = null;

    this.init();
  }

  /**
   * Log debug messages if debug mode is enabled
   */
  logDebug(...args) {
    if (this.debug) {
      console.debug(...args);
    }
  }

  /**
   * Initialize the SelectChoice component
   */
  init() {
    const selectElement = document.getElementById(this.elementId);

    if (!selectElement) {
      console.warn(
        `SelectChoice: Element with ID '${this.elementId}' not found`
      );
      return;
    }

    if (typeof window.Choices === "undefined") {
      console.warn("SelectChoice: Choices library not found");
      return;
    }

    // Initialize Choices with the configuration
    this.choicesElement = new window.Choices(selectElement, this.choicesConfig);

    if (this.choicesElement) {
      this.bindEvents();
      this.setupLivewireListeners();

      // Mark initialization as complete after a short delay
      setTimeout(() => {
        this.isInitializing = false;
      }, 100);
    }
  }

  /**
   * Bind event listeners to the Choices element
   */
  bindEvents() {
    if (!this.choicesElement) return;

    const selectElement = document.getElementById(this.elementId);
    if (selectElement) {
      selectElement.addEventListener("change", (event) => {
        this.handleSelectionChange(event);
      });
    }
  }

  /**
   * Handle selection change events
   */
  handleSelectionChange(event) {
    // Skip dispatching during initialization
    if (this.isInitializing) return;

    const selectedValues = this.choicesElement.getValue(true);
    const value = this.isMultiple ? selectedValues : selectedValues[0] || null;

    this.logDebug("SelectChoice Selection Changed:", {
      id: this.elementId,
      values: selectedValues,
      value: value,
      isMultiple: this.isMultiple,
    });

    if (window.Livewire) {
      if (this.dispatchGenericEvent) {
        // Dispatch generic event with id in payload
        window.Livewire.dispatch("SelectChoiceUpdated", {
          value: value,
          id: this.elementId,
        });
      } else {
        // Dispatch specific event (default behavior)
        window.Livewire.dispatch(`SelectChoiceUpdated:${this.elementId}`, {
          value: value,
        });
      }
    }
  }

  /**
   * Setup Livewire event listeners
   */
  setupLivewireListeners() {
    if (!window.Livewire) return;

    // Reset all selections on Livewire event
    Livewire.on(`SelectChoiceReset:${this.elementId}`, () => {
      this.reset();
    });

    // Remove specific values from selection on Livewire event
    Livewire.on(`SelectChoiceRemoveValue:${this.elementId}`, ({ value }) => {
      this.removeValue(value);
    });
  }

  /**
   * Reset the selection
   */
  reset() {
    this.logDebug(`SelectChoice Reset triggered for ${this.elementId}`);

    if (this.choicesElement && this.choicesElement.removeActiveItems) {
      this.choicesElement.removeActiveItems();
    }
  }

  /**
   * Remove specific values from the selection
   */
  removeValue(value) {
    this.logDebug(
      `SelectChoice RemoveValue triggered for ${this.elementId}:`,
      value
    );

    if (!value || !this.choicesElement) return;

    try {
      if (Array.isArray(value)) {
        // Remove multiple values
        value.forEach((val) => {
          if (this.choicesElement.removeActiveItemsByValue) {
            this.choicesElement.removeActiveItemsByValue(val);
          }
        });
      } else {
        // Remove single value
        if (this.choicesElement.removeActiveItemsByValue) {
          this.choicesElement.removeActiveItemsByValue(value);
        }
      }
    } catch (error) {
      console.error("Error removing SelectChoice values:", error);
    }
  }

  /**
   * Set the value of the SelectChoice
   */
  setValue(value) {
    if (!this.choicesElement) return;

    try {
      if (this.choicesElement.setChoiceByValue) {
        this.choicesElement.setChoiceByValue(value);
      }
    } catch (error) {
      console.error("Error setting SelectChoice value:", error);
    }
  }

  /**
   * Get the current value of the SelectChoice
   */
  getValue() {
    if (!this.choicesElement) return null;

    const selectedValues = this.choicesElement.getValue(true);
    return this.isMultiple ? selectedValues : selectedValues[0] || null;
  }

  /**
   * Destroy the SelectChoice instance
   */
  destroy() {
    if (this.choicesElement) {
      // Remove event listeners if needed
      const selectElement = document.getElementById(this.elementId);
      if (selectElement) {
        selectElement.removeEventListener("change", this.handleSelectionChange);
      }

      // Destroy Choices instance if it has a destroy method
      if (this.choicesElement.destroy) {
        this.choicesElement.destroy();
      }
    }
  }

  /**
   * Static method to create and initialize a SelectChoice instance
   */
  static create(
    elementId,
    choicesConfig = {},
    isMultiple = false,
    dispatchGenericEvent = false,
    debug = false
  ) {
    return new SelectChoice(
      elementId,
      choicesConfig,
      isMultiple,
      dispatchGenericEvent,
      debug
    );
  }
}
