# Form Components based on Bulma

A package for components based on Laravel and Bulma. This package provides a comprehensive set of form components with built-in validation, styling, and enhanced functionality.

## Overview

- Built on Laravel's component system
- Integrates with Bulma CSS framework
- Includes advanced features like file uploads, date picking, and dynamic selects
- Supports form validation and custom rules
- Responsive and accessible form elements

## Dependencies

- [Laravel HTML](https://spatie.be/docs/laravel-html/v3/introduction)
- [jQuery](https://jquery.com/)
- [Bulma Calender](https://bulma-calendar.onrender.com/)
- [Bulma TagsInput](https://bulma-tagsinput.netlify.app/)
- [Cropper.js](https://github.com/fengyuanchen/cropperjs) For ImagePicker
- [WebcamJS](https://github.com/jhuckaby/webcamjs/) - For ImagePicker
- [flatpickr](https://flatpickr.js.org/) - For datetime picker
- [Filepond](https://pqina.nl/filepond/) - For File upload
- [Quill](https://quilljs.com/docs/quickstart) - For Rich text editor

## Installation

```
$ composer require v360-tech/components-form
```

In case your Laravel version does NOT autoload the packages, add the service provider to config/app.php:

```
V360\FormComponents\FormComponentsServiceProvider::class,
```

`For image picker, Cropper object should be available on client side. So need to follow installation steps as per auther's document.`

## Export

### To publish views

```
php artisan vendor:publish --provider="V360\FormComponents\FormComponentsServiceProvider" --tag="views"
```

### To publish configs

`Required in case of ReCaptcha component.`

```
php artisan vendor:publish --provider="V360\FormComponents\FormComponentsServiceProvider" --tag="config"
```

## Component Documentation

This package implements a comprehensive set of form components. Each component is designed to be easily integrated into your Laravel application with minimal configuration.

### Basic Input Components

| Component | Tag                    | Description                 | Key Features                                              |
| --------- | ---------------------- | --------------------------- | --------------------------------------------------------- |
| Text      | `<x-vform-text />`     | Basic text input field      | - Label support<br>- Placeholder text<br>- Custom classes |
| Number    | `<x-vform-number />`   | Numeric input field         | - Min/max values<br>- Step control<br>- Right alignment   |
| Email     | `<x-vform-email />`    | Email input with validation | - Built-in email validation<br>- Custom error messages    |
| Password  | `<x-vform-password />` | Secure password input       | - Password strength indicator<br>- Show/hide toggle       |
| Textarea  | `<x-vform-textarea />` | Multi-line text input       | - Resizable<br>- Auto-grow option                         |

### Advanced Input Components

| Component      | Tag                          | Description            | Key Features                                                                      |
| -------------- | ---------------------------- | ---------------------- | --------------------------------------------------------------------------------- |
| Amount         | `<x-vform-amount />`         | Currency input field   | - Multiple currencies (₹,$,€,£)<br>- Right alignment<br>- Automatic formatting    |
| Datepicker     | `<x-vform-datepicker />`     | Date/time selection    | - Date/Time/DateTime modes<br>- Min/max date limits<br>- Custom time restrictions |
| Autocomplete   | `<x-vform-autocomplete />`   | Input with suggestions | - Dynamic suggestions<br>- Custom data source<br>- Debounced search               |
| Dynamic Select | `<x-vform-dynamic-select />` | AJAX-powered select    | - Remote data loading<br>- Search functionality<br>- Pagination support           |
| Tags           | `<x-vform-tags />`           | Tag input field        | - Multiple tags support<br>- Custom validation<br>- Auto-completion               |

### File and Media Components

| Component    | Tag                        | Description               | Key Features                                                  |
| ------------ | -------------------------- | ------------------------- | ------------------------------------------------------------- |
| File         | `<x-vform-file />`         | Basic file upload         | - Multiple file support<br>- File type validation             |
| FilePond     | `<x-vform-file-pond />`    | Advanced file upload      | - Drag & drop<br>- Image preview<br>- Upload progress         |
| Image Picker | `<x-vform-image-picker />` | Image selection/capture   | - Camera integration<br>- Image cropping<br>- Preview support |
| Media Select | `<x-vform-media-select />` | Media library integration | - Gallery view<br>- Multiple selection<br>- Filterable        |

### Selection Components

| Component | Tag                    | Description                | Key Features                                               |
| --------- | ---------------------- | -------------------------- | ---------------------------------------------------------- |
| Checkbox  | `<x-vform-checkbox />` | Single/multiple checkboxes | - Group support<br>- Custom styling                        |
| Radio     | `<x-vform-radio />`    | Radio button group         | - Horizontal/Vertical layout<br>- Custom labels            |
| Select    | `<x-vform-select />`   | Dropdown selection         | - Option groups<br>- Multiple selection<br>- Search filter |

### Special Components

| Component | Tag                      | Description                  | Key Features                                                     |
| --------- | ------------------------ | ---------------------------- | ---------------------------------------------------------------- |
| ReCaptcha | `<x-vform-re-captcha />` | Google ReCaptcha integration | - v2/v3 support<br>- Custom validation<br>- Error handling       |
| Table     | `<x-vform-table />`      | Dynamic table input          | - Add/Remove rows<br>- Column configuration<br>- Data validation |
| Keyvalue  | `<x-vform-keyvalue />`   | Key-value pair input         | - Dynamic pairs<br>- Custom validation<br>- Sortable entries     |

### Other components

| Element       | Tag                       |
| :------------ | ------------------------- |
| A Date Picker | \<x-vform-a-datepicker /> |

## Component Usage Examples

### Basic Components

```blade
{{-- Text Input --}}
<x-vform-text
    name="username"
    label="Username"
    placeholder="Enter username"
    :hint="'Must be unique'"
/>

{{-- Amount Input --}}
<x-vform-amount
    name="price"
    currency="dollar"
    :value="19.99"
    label="Price"
/>

{{-- Date Picker --}}
<x-vform-datepicker
    name="event_date"
    type="datetime"
    :dateLimits="['min' => now(), 'max' => now()->addYear()]"
    label="Event Date"
/>
```

### Advanced Components

```blade
{{-- Dynamic Select --}}
<x-vform-dynamic-select
    name="category"
    :url="route('api.categories')"
    label="Category"
    placeholder="Select a category"
/>

{{-- File Upload with FilePond --}}
<x-vform-file-pond
    name="documents[]"
    multiple
    accept="application/pdf"
    :maxFileSize="5000000"
    label="Upload Documents"
/>

{{-- Key-Value Pairs --}}
<x-vform-keyvalue
    name="meta_data"
    :value="$existingData"
    label="Meta Data"
/>
```

### Form Validation Components

```blade
{{-- ReCaptcha Integration --}}
<x-vform-re-captcha />

{{-- Controller Validation --}}
use V360\FormComponents\Rules\ReCaptcha;

public function store(Request $request)
{
    $inputs = [
        'username' => ['required', 'min:3'],
        'email' => ['required', 'email'],
        'g-recaptcha-response' => ['required', new ReCaptcha],
    ];

    $messages = [
        'g-recaptcha-response.required' => 'Please confirm that you are not a robot!',
    ];

    $this->validate($request, $inputs, $messages);

    // Process the form...
}
```

### JavaScript Integration

Some components require JavaScript initialization. Make sure to include the necessary dependencies and initialize them properly:

```html
{{-- Include Required Scripts --}}
<script src="https://unpkg.com/filepond"></script>
<script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>

{{-- Initialize Components --}}
<script>
  document.addEventListener("DOMContentLoaded", function () {
    // Initialize datepickers
    flatpickr(".datepicker", {
      enableTime: true,
      dateFormat: "Y-m-d H:i",
    });

    // Initialize FilePond
    FilePond.create(document.querySelector('input[type="file"]'));
  });
</script>
```

### Styling

Components use Bulma classes by default, but you can customize the appearance:

```blade
<x-vform-text
    name="custom_input"
    :classes="'is-large is-rounded custom-class'"
    label="Custom Styled Input"
/>
```

For more detailed documentation about each component and its available options, please refer to the individual component documentation in the `docs/` directory.

### Documentation

In order to generate documentation for this package run the following command.

```
phpdoc -d . -t ./docs
```
