<?php

namespace V360\FormComponents\View\Components;

use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Illuminate\View\Component;

/**
 * Table Laravel Blade Component
 *
 * A dynamic table component allowing users to add or remove rows.
 * Features customizable fields, labels, and sample data for enhanced user experience.
 *
 * Features:
 * - Dynamic row addition and removal
 * - Customizable fields with various input types
 * - Sample data display for guidance
 * - Customizable labels
 * - Form validation integration
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-08-01
 *
 * @example
 * <!-- Dynamic table component -->
 * <x-vform-table 
 *     name="items" 
 *    :fields="[['text', 'item_name', 'Item Name'], ['number', 'quantity', 'Quantity']]" 
 *     :widths="['70%', '30%']"
 *    :required="['item_name']"
 *    :defaults="['quantity' => 1]"
 *     label="Order Items"
 *    sample="true"
 * />
 *
 */
class Table extends Component
{

  /**
   * Collection of fields
   *
   * @var Collection
   */
  public string $name;
  public Collection $fields;
  public array $widths;
  public string $label;
  public string $id;
  public bool $sample;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param array $fields - Array of fields [type (text|number), name, label]
   * @param array $widths - Array of field widths
   * @param array $required - Array of required fields
   * @param array $defaults - Key-value pair of default value of each field 
   * @param array $min - Key-value pair of min value of each field 
   * @param array $max - Key-value pair of max value of each field 
   * @param array $step - Key-value pair of step value of each field 
   * @param array $pattern - Key-value pair of pattern for each field   * 
   * @param string|null $label
   * @param bool $sample - Whether to show sample data or not
   */
  public function __construct(
    string $name,
    array $fields,
    array $widths = [],
    array $required = [],
    array $defaults = [],
    array $min = [],
    array $max = [],
    array $step = [],
    array $pattern = [],
    ?string $label = null,
    bool $sample = true
  ) {
    $this->name = $name;
    $this->id = Str::uuid();
    $this->label = $label ?? Str::headline($name);
    $this->fields = collect();
    $this->widths = $widths;

    //prepare fields 
    foreach ($fields as $field) {
      [$type, $name, $label] = $field;
      $input = [
        'type' => $type,
        'name' => $name,
        'label' => $label,
        'placeholder' => $field[3] ?? $label
      ];
      $input['required'] = in_array($name, $required) ? 'required' : '';
      if ($type == 'number') {
        $input['value'] = isset($defaults[$name]) ? "value={$min[$name]}" : null;
        $input['min'] = isset($min[$name]) ? "min={$min[$name]}" : null;
        $input['max'] = isset($max[$name]) ? "max={$max[$name]}" : null;
        $input['step'] = $step[$name] ?? 1;
      }
      $input['pattern'] = isset($pattern[$name]) ? "pattern={$pattern[$name]}" : null;
      $this->fields->push($input);
    }
    $this->sample = $sample;
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('vform::components.table');
  }
}
