<?php

namespace V360\FormComponents\View\Components;

use Illuminate\View\Component;
use Illuminate\Support\Str;
use Illuminate\Support\Collection;

/**
 * Keyvalue Laravel Blade Component
 *
 * A key-value pair input component that allows users to add, edit, and remove key-value pairs.
 * Features customizable labels, hints, and placeholders for enhanced user experience.
 *
 * Features:
 * - Dynamic addition and removal of key-value pairs
 * - Customizable labels and placeholders
 * - Form validation integration
 * - Bulma CSS framework integration
 * - Mobile-friendly responsive design
 *
 * @package V360\FormComponents\View\Components
 * @author V360
 * @version 1.0.0
 * @since 2024-07-20
 *
 * @example
 * <!-- Key-value pair input -->
 * <x-vform-keyvalue 
 *     name="settings" 
 *     :value="['theme' => 'dark', 'notifications' => 'enabled']" 
 *     label="Settings" 
 *     placeholderKey="Enter Key" 
 *     placeholderValue="Enter Value" 
 *     hint="Add your settings as key-value pairs." 
 * />
 *
 */
class Keyvalue extends Component
{
  public string $name;
  public $value;
  public Collection $values;
  public ?string $label = null;
  public string $id;
  public string $placeholderKey;
  public string $placeholderValue;
  public ?string $hint = null;

  /**
   * Type of value filed is number or not
   *
   * @var boolean
   */
  public bool $isNumber = false;

  /**
   * Value is required or not
   *
   * @var boolean
   */
  public bool $valueRequired = true;

  /**
   * Create a new component instance
   *
   * @param string $name
   * @param mixed $value
   * @param string|null $id
   * @param mixed $label
   * @param array $attributes
   * @param string|null $placeholderKey
   * @param string|null $placeholderValue
   * @param string|null $hint
   * @param bool $isNumber
   * @param bool $valueRequired
   */
  public function __construct(
    string $name,
    mixed $value = null,
    ?string $id = null,
    mixed $label = null,
    array $attributes = [],
    ?string $placeholderKey = null,
    ?string $placeholderValue = null,
    ?string $hint = null,
    bool $isNumber = false,
    bool $valueRequired = true
  ) {
    $this->name = $name;
    $this->id = $id ? $id : str_replace('_', '-', $name);
    $this->label = $label ? $label : ($label === false ? false : Str::title(str_replace('_', ' ', $name)));
    $this->attr = array_merge(['id' => $this->id, 'class' => 'input'], $attributes);
    $this->placeholderKey = Str::title(str_replace('_', ' ', $placeholderKey));
    $this->placeholderValue = Str::title(str_replace('_', ' ', $placeholderValue));
    if ($value) {
      $this->value = json_encode($value);
      $this->values = collect($value);
    } else {
      $this->values = collect();
    }
    $this->hint = $hint;
    $this->isNumber = $isNumber;
    $this->valueRequired = $valueRequired;
  }

  /**
   * Get the view / contents that represent the component.
   *
   * @return \Illuminate\Contracts\View\View|string
   */
  public function render()
  {
    return view('vform::components.keyvalue');
  }
}
