/**
 * RangeSelector JavaScript Class
 * Handles the initialization and management of noUiSlider instances
 */
export default class RangeSelector {
  constructor(sliderId, config, options) {
    this.sliderId = sliderId;
    this.slider = document.getElementById(sliderId);
    this.config = config;
    this.options = options;
    this.inputName = this.slider?.dataset.inputName;
    this.isRange = this.slider?.dataset.range === "true";
    this.showPips = this.slider?.dataset.showPips === "true";
    this.showInput = this.slider?.dataset.showInput === "true";
    this.step = parseFloat(this.slider?.dataset.step) || 1;
    this.debug = config.debug || false;

    this.init();
  }

  /**
   * Log debug messages if debug mode is enabled
   */
  logDebug(...args) {
    if (this.debug) {
      console.debug(...args);
    }
  }

  /**
   * Initialize the slider if element and noUiSlider are available
   */
  init() {
    if (!this.slider || !window.noUiSlider) {
      console.warn(
        "RangeSelector: Missing slider element or noUiSlider library"
      );
      return;
    }

    this.setupPipsForOptions();
    this.createSlider();
    this.bindEvents();
    this.initInputFields();
  }

  /**
   * Setup pips configuration for options-based sliders
   */
  setupPipsForOptions() {
    if (!this.showPips) {
      return;
    }
    if (this.options.length > 0) {
      this.config.pips = {
        mode: "steps",
        stepped: true,
        density: 100 / (this.options.length - 1),
        format: {
          to: (value) => {
            const index = Math.round(value);
            return this.options[index] !== undefined
              ? this.options[index]
              : value;
          },
          from: (value) => {
            return this.options.indexOf(value);
          },
        },
      };
    }
  }

  /**
   * Create the noUiSlider instance
   */
  createSlider() {
    try {
      noUiSlider.create(this.slider, this.config);
      this.logDebug(
        "RangeSelector: Slider created successfully",
        this.sliderId
      );
    } catch (error) {
      console.error("RangeSelector: Failed to create slider", error);
    }
  }

  /**
   * Bind slider update events
   */
  bindEvents() {
    this.slider.noUiSlider.on("update", (values, handle) => {
      const data = this.processSliderValues(values);
      this.dispatchUpdateEvent(data, values, handle);
      this.updateInputFields(values, handle);
    });

    this.setupLivewireListeners();
  }

  /**
   * Setup Livewire event listeners
   */
  setupLivewireListeners() {
    if (!window.Livewire) return;

    // Listen for reset event with names array
    Livewire.on("RangeSelectorReset", ({ names }) => {
      if (names && names.includes(this.inputName)) {
        this.reset();
      }
    });

    // Listen for set event specific to this component
    Livewire.on(`RangeSelectorSet:${this.inputName}`, ({ value }) => {
      this.setValue(value);
    });
  }

  /**
   * Reset the slider to its initial values
   */
  reset() {
    this.logDebug(`RangeSelector Reset triggered for ${this.inputName}`);

    if (this.slider.noUiSlider) {
      // config.start is already an array: [start] for single value, [start, end] for range
      const initialValues = this.config.start;
      this.logDebug("Resetting to initial values:", initialValues);
      this.slider.noUiSlider.set(initialValues);
    }
  }

  /**
   * Set specific value(s) for the slider
   * @param {number|Array} value - New value(s) to set
   */
  setValue(value) {
    this.logDebug(
      `RangeSelector SetValue triggered for ${this.inputName}:`,
      value
    );

    if (value === null || value === undefined || !this.slider.noUiSlider)
      return;

    try {
      // Ensure value is in the correct format for noUiSlider
      let normalizedValue = value;
      if (this.isRange && !Array.isArray(value)) {
        // If it's a range slider but single value provided, use current end value
        const currentValues = this.slider.noUiSlider.get();
        normalizedValue = [value, currentValues[1]];
      } else if (!this.isRange && Array.isArray(value)) {
        // If it's single slider but array provided, use first value
        normalizedValue = value[0];
      }

      this.slider.noUiSlider.set(normalizedValue);
    } catch (error) {
      console.error("Error setting RangeSelector values:", error);
    }
  }

  /**
   * Process slider values based on range mode and options
   * @param {Array} values - Raw slider values
   * @returns {Object} Processed data object
   */
  processSliderValues(values) {
    const data = {};

    if (this.isRange) {
      data.min = parseFloat(values[0]);
      data.max = parseFloat(values[1]);

      if (this.options.length > 0) {
        const minIndex = Math.round(values[0]);
        const maxIndex = Math.round(values[1]);
        data.values = this.options.slice(minIndex, maxIndex + 1);
      }
    } else {
      data.value = parseFloat(values[0]);

      if (this.options.length > 0) {
        const index = Math.round(values[0]);
        data.selectedOption = this.options[index];
      }
    }

    return data;
  }

  /**
   * Dispatch update events to Livewire and custom listeners
   * @param {Object} data - Processed slider data
   * @param {Array} values - Raw slider values
   * @param {number} handle - Handle index that triggered the update
   */
  dispatchUpdateEvent(data, values, handle) {
    this.logDebug("RangeSelector: Slider Updated", {
      sliderId: this.sliderId,
      values,
      handle,
      data,
    });

    // Dispatch to Livewire if available
    if (window.Livewire) {
      this.logDebug(
        "RangeSelector: Dispatching Livewire event",
        this.inputName,
        data
      );
      Livewire.dispatch(`RangeSelectorUpdated:${this.inputName}`, {
        data,
      });
      return;
    }

    // Dispatch custom event for non-Livewire listeners
    const customEvent = new CustomEvent(
      `RangeSelectorUpdated:${this.inputName}`,
      {
        detail: {
          data,
          values,
          handle,
          sliderId: this.sliderId,
        },
      }
    );
    window.dispatchEvent(customEvent);
  }

  /**
   * Public method to update slider values programmatically
   * @param {number|Array} values - New value(s) to set
   */
  updateValues(values) {
    if (this.slider.noUiSlider) {
      this.slider.noUiSlider.set(values);
    }
  }

  /**
   * Public method to destroy the slider instance
   */
  destroy() {
    if (this.slider.noUiSlider) {
      this.slider.noUiSlider.destroy();
      this.logDebug("RangeSelector: Slider destroyed", this.sliderId);
    }
  }

  /**
   * Initialize input fields for direct value entry
   */
  initInputFields() {
    if (!this.showInput) {
      return;
    }

    this.startInput = document.getElementById(`${this.sliderId}_start_input`);
    this.endInput = document.getElementById(`${this.sliderId}_end_input`);

    this.bindInputEvents();
  }

  /**
   * Bind input field events to update slider
   */
  bindInputEvents() {
    if (this.startInput) {
      this.startInput.addEventListener("blur", (e) => {
        this.handleInputChange(e.target, 0);
      });

      this.startInput.addEventListener("focus", (e) => {
        e.target.select();
      });
    }

    if (this.endInput) {
      this.endInput.addEventListener("blur", (e) => {
        this.handleInputChange(e.target, 1);
      });

      this.endInput.addEventListener("focus", (e) => {
        e.target.select();
      });
    }
  }

  /**
   * Handle input field changes and update slider
   * @param {HTMLInputElement} input - The input element that changed
   * @param {number} handle - Handle index (0 for start, 1 for end)
   */
  handleInputChange(input, handle) {
    const value = parseFloat(input.value);
    this.logDebug("Input Change Detected:", {
      value,
      handle,
    });

    if (isNaN(value)) {
      return;
    }

    if (this.isRange) {
      const currentValues = this.slider.noUiSlider.get();
      const newValues = [...currentValues];
      newValues[handle] = value;
      this.slider.noUiSlider.set(newValues);
    } else {
      this.slider.noUiSlider.set(value);
    }
  }

  /**
   * Update input field values when slider changes
   * @param {Array} values - Current slider values
   * @param {number} handle - Handle index that triggered the update
   */
  updateInputFields(values, handle) {
    if (!this.showInput) {
      return;
    }

    const decimals = this.step < 1 ? 2 : 0;

    if (this.isRange) {
      if (handle === 0 && this.startInput) {
        this.startInput.value = parseFloat(values[0]).toFixed(decimals);
      } else if (handle === 1 && this.endInput) {
        this.endInput.value = parseFloat(values[1]).toFixed(decimals);
      }
    } else if (this.startInput) {
      this.startInput.value = parseFloat(values[0]).toFixed(decimals);
    }
  }

  /**
   * Static method to create and initialize a RangeSelector instance
   */
  static create(sliderId, config, options) {
    return new RangeSelector(sliderId, config, options);
  }
}
