/**
 * RangeSelector JavaScript Class
 * Handles the initialization and management of noUiSlider instances
 */
export default class RangeSelector {
  constructor(sliderId, config, options) {
    this.sliderId = sliderId;
    this.slider = document.getElementById(sliderId);
    this.config = config;
    this.options = options;
    this.inputName = this.slider?.dataset.inputName;
    this.isRange = this.slider?.dataset.range === "true";
    this.showPips = this.slider?.dataset.showPips === "true";
    this.debug = config.debug || false;

    this.init();
  }

  /**
   * Log debug messages if debug mode is enabled
   */
  logDebug(...args) {
    if (this.debug) {
      console.debug(...args);
    }
  }

  /**
   * Initialize the slider if element and noUiSlider are available
   */
  init() {
    if (!this.slider || !window.noUiSlider) {
      console.warn(
        "RangeSelector: Missing slider element or noUiSlider library"
      );
      return;
    }

    this.setupPipsForOptions();
    this.createSlider();
    this.bindEvents();
  }

  /**
   * Setup pips configuration for options-based sliders
   */
  setupPipsForOptions() {
    if (!this.showPips) {
      return;
    }
    if (this.options.length > 0) {
      this.config.pips = {
        mode: "steps",
        stepped: true,
        density: 100 / (this.options.length - 1),
        format: {
          to: (value) => {
            const index = Math.round(value);
            return this.options[index] !== undefined
              ? this.options[index]
              : value;
          },
          from: (value) => {
            return this.options.indexOf(value);
          },
        },
      };
    }
  }

  /**
   * Create the noUiSlider instance
   */
  createSlider() {
    try {
      noUiSlider.create(this.slider, this.config);
      this.logDebug(
        "RangeSelector: Slider created successfully",
        this.sliderId
      );
    } catch (error) {
      console.error("RangeSelector: Failed to create slider", error);
    }
  }

  /**
   * Bind slider update events
   */
  bindEvents() {
    this.slider.noUiSlider.on("update", (values, handle) => {
      const data = this.processSliderValues(values);
      this.dispatchUpdateEvent(data, values, handle);
    });
  }

  /**
   * Process slider values based on range mode and options
   * @param {Array} values - Raw slider values
   * @returns {Object} Processed data object
   */
  processSliderValues(values) {
    const data = {};

    if (this.isRange) {
      data.min = parseFloat(values[0]);
      data.max = parseFloat(values[1]);

      if (this.options.length > 0) {
        const minIndex = Math.round(values[0]);
        const maxIndex = Math.round(values[1]);
        data.values = this.options.slice(minIndex, maxIndex + 1);
      }
    } else {
      data.value = parseFloat(values[0]);

      if (this.options.length > 0) {
        const index = Math.round(values[0]);
        data.selectedOption = this.options[index];
      }
    }

    return data;
  }

  /**
   * Dispatch update events to Livewire and custom listeners
   * @param {Object} data - Processed slider data
   * @param {Array} values - Raw slider values
   * @param {number} handle - Handle index that triggered the update
   */
  dispatchUpdateEvent(data, values, handle) {
    this.logDebug("RangeSelector: Slider Updated", {
      sliderId: this.sliderId,
      values,
      handle,
      data,
    });

    // Dispatch to Livewire if available
    if (window.Livewire) {
      this.logDebug(
        "RangeSelector: Dispatching Livewire event",
        this.inputName,
        data
      );
      Livewire.dispatch(`RangeSelectorUpdated:${this.inputName}`, {
        data,
      });
      return;
    }

    // Dispatch custom event for non-Livewire listeners
    const customEvent = new CustomEvent(
      `RangeSelectorUpdated:${this.inputName}`,
      {
        detail: {
          data,
          values,
          handle,
          sliderId: this.sliderId,
        },
      }
    );
    window.dispatchEvent(customEvent);
  }

  /**
   * Public method to update slider values programmatically
   * @param {number|Array} values - New value(s) to set
   */
  updateValues(values) {
    if (this.slider.noUiSlider) {
      this.slider.noUiSlider.set(values);
    }
  }

  /**
   * Public method to destroy the slider instance
   */
  destroy() {
    if (this.slider.noUiSlider) {
      this.slider.noUiSlider.destroy();
      this.logDebug("RangeSelector: Slider destroyed", this.sliderId);
    }
  }
}
