<?php

namespace V360\ActionComponents\View\Components;

use Closure;
use Illuminate\Contracts\View\View;
use Illuminate\View\Component;
use Illuminate\Support\Str;

/**
 * Button Scroll Up Component
 * 
 * A Laravel Blade component that renders a scroll-up button that allows users
 * to smoothly scroll back to the top of the page or to a specific target element.
 * The button is positioned as a floating action button with customizable styling
 * and positioning options.
 * 
 * @package V360\ActionComponents\View\Components
 * @author  V360 Action Components Package
 * @version 1.0.0
 * @since   1.0.0
 * 
 * @property string|null $id       Unique identifier for the scroll-up button (auto-generated UUID)
 * @property string|null $target   CSS selector or element ID to scroll to (default: null for page top)
 * @property int         $zIndex   CSS z-index value for button positioning (default: 1)
 * @property int         $size     Button size in pixels (default: 64)
 * @property string      $color    Button color scheme/theme (default: 'link')
 * @property string      $position Button position on screen: 'left' or 'right' (default: 'right')
 * @property string      $title    Tooltip text displayed on hover (default: 'Scroll to top')
 */
class ButtonScrollUp extends Component
{
  public $id;

  /**
   * Create a new ButtonScrollUp component instance.
   * 
   * Initializes the scroll-up button component with customizable properties for
   * target scrolling, styling, and positioning. Automatically generates a unique
   * ID for the button instance.
   *
   * @param string|null $target   The CSS selector or element ID to scroll to. 
   *                              If null, scrolls to the top of the page.
   *                              Examples: '#header', '.top-section', null
   * @param int        $zIndex    The CSS z-index value for button layering.
   *                              Higher values bring the button to the front.
   *                              Default: 1
   * @param int        $size      The button size in pixels (width and height).
   *                              Default: 64
   * @param string     $color     The color scheme/theme for the button.
   *                              Available options depend on your CSS framework.
   *                              Default: 'link'
   * @param string     $position  The screen position for the button.
   *                              Accepts: 'left' | 'right'
   *                              Default: 'right'
   * @param string     $title     The tooltip text displayed on hover.
   *                              Default: 'Scroll to top'
   * 
   * @return void
   * 
   * @throws \InvalidArgumentException If position is not 'left' or 'right'
   * 
   * @example
   * <!-- Basic usage - scroll to page top -->
   * <x-vaction::button-scroll-up />
   * 
   * @example
   * <!-- Scroll to specific element with custom styling -->
   * <x-vaction::button-scroll-up 
   *     target="#header"
   *     :z-index="999"
   *     :size="48"
   *     color="primary"
   *     position="right" />
   * 
   * @example
   * <!-- With dynamic values from variables -->
   * <x-vaction::button-scroll-up 
   *     :target="$scrollTarget"
   *     :z-index="$zIndex"
   *     :size="$buttonSize"
   *     :color="$theme"
   *     :position="$buttonPosition" />
   */
  public function __construct(
    public ?string $target = null,
    public int $zIndex = 4,
    public int $size = 64,
    public string $color = 'link',
    public string $position = 'right',
    public string $title = 'Scroll to top',
  ) {
    //
    $this->id = Str::uuid();
    //validate position
    $validPositions = ['left', 'right'];
    if (!\in_array($this->position, $validPositions)) {
      throw new \InvalidArgumentException("Invalid position value: {$this->position}. Allowed values are 'left' or 'right'.");
    }
  }

  /**
   * Get the view / contents that represent the component.
   * 
   * @return View|Closure|string The rendered view instance, closure, or string content
   *                            for the scroll-up button component
   * @see resources/views/components/button-scroll-up.blade.php The corresponding Blade template
   */
  public function render(): View|Closure|string
  {
    return view('vaction::components.button-scroll-up');
  }
}
