# Action Components

A Laravel package for UI action components based on the Bulma CSS framework. This package provides a collection of reusable Laravel Blade components that simplify the implementation of common UI actions and interfaces.

## Version

Current Version: 3.1.0

## Dependencies

- PHP ^7.3|^8.0
- [Laravel](https://laravel.com/) (with Laravel Blade Components support)
- [jQuery](https://jquery.com/)
- [Bulma CSS Framework](https://bulma.io/)
- [Material Symbols](https://fonts.google.com/icons) (for icons)
- [BulmaQuickView](https://wikiki.github.io/components/quickview/)

## Installation

Install the package via Composer:

```bash
$ composer require v360-tech/components-action
```

Laravel's package auto-discovery feature will automatically register the service provider. If you're using an older version of Laravel that doesn't support auto-discovery, add the service provider manually to your `config/app.php`:

```php
'providers' => [
    // ...
    V360\ActionComponents\ActionComponentsServiceProvider::class,
],
```

## Publish Assets

### Publish Views

To customize the component views, you can publish them to your project:

```bash
php artisan vendor:publish --provider="V360\ActionComponents\ActionComponentsServiceProvider" --tag="vaction"
```

This will copy all component views to your `resources/views/vendor/v360/action-components` directory.

## Usage

This package provides several components that can be used in your Blade templates. Each component is designed to handle a specific UI action pattern.

| Component          | Tag                                        | Description                                                                   |
| :----------------- | ------------------------------------------ | ----------------------------------------------------------------------------- |
| Bulk Action        | `<x-vaction-bulkx />`                      | Perform actions on multiple selected records                                  |
| Button             | `<x-vaction-button />`                     | Base button component with customizable options                               |
| Button Close       | `<x-vaction-button-close />`               | Button with close icon for dismissing elements                                |
| Button Delete      | `<x-vaction-button-delete />`              | Delete/archive button with confirmation dialog                                |
| Button Edit        | `<x-vaction-button-edit />`                | Edit action button with edit icon                                             |
| Button Icon        | `<x-vaction-button-icon />`                | Icon-only button with customizable Material icon                              |
| Button Show        | `<x-vaction-button-show />`                | Information/details button with info icon                                     |
| Button Scroll Up   | `<x-vaction-button-scroll-up />`           | Floating scroll-up button that smoothly scrolls to page top or target element |
| Card List          | `<x-vaction-card-list />`                  | Display items in a card-based grid layout                                     |
| Collapsible Filter | `<x-vaction-collapsible-filter />`         | Collapsible panel with filtering options                                      |
| Filter Actions     | `<x-vaction-collapsible-filter-actions />` | Actions for filter panel                                                      |
| Default Actions    | `<x-vaction-defaultx />`                   | Standard action buttons with configurable options                             |
| Get Request        | `<x-vaction-getx />`                       | Button that triggers a GET request                                            |
| Infinite List      | `<x-vaction-infinite-list />`              | Lazy-loading list with infinite scroll                                        |
| Modal Edit         | `<x-vaction-modal-edit />`                 | Edit form in a modal popup                                                    |
| Post Request       | `<x-vaction-postx />`                      | Button that triggers a POST request                                           |
| Quick Filter       | `<x-vaction-quick-filter />`               | Compact filtering controls                                                    |
| Quick Form         | `<x-vaction-quick-form />`                 | Simple, compact form                                                          |
| Save Cancel        | `<x-vaction-save-cancel />`                | Save and cancel action buttons for forms                                      |

### Component Examples

Below are examples of how to use some of the key components:

#### Bulk Actions

```blade
<x-vaction-bulkx
    route-prefix="admin.users"
    :actions="['approve' => 'Approve Selected', 'delete' => 'Delete Selected']"
    record-table-id="#users-table"
/>
```

#### Get Request Button

```blade
<x-vaction-getx
    route="{{ route('users.show', $user->id) }}"
    icon="visibility"
    color="info"
    title="View"
    tooltip="View user details"
/>
```

#### Post Request Button

```blade
<x-vaction-postx
    route="{{ route('users.approve', $user->id) }}"
    icon="check_circle"
    color="success"
    title="Approve"
    confirm="Are you sure you want to approve this user?"
/>
```

#### Modal Edit

```blade
<x-vaction-modal-edit
    route="{{ route('users.edit', $user->id) }}"
    icon="edit"
    color="warning"
    title="Edit"
    modal-title="Edit User"
/>
```

#### Infinite List

```blade
<x-vaction-infinite-list
    load-more-url="{{ route('api.users.index') }}"
    container-id="users-container"
    :params="['status' => 'active']"
/>
```

#### Card List

```blade
<x-vaction-card-list
    :items="$products"
    card-class="is-fullwidth"
>
    @slot('header', $product)
        <div class="card-header-title">
            {{ $product->name }}
        </div>
    @endslot

    @slot('content', $product)
        <div class="content">
            {{ $product->description }}
            <p class="has-text-weight-bold">Price: ${{ $product->price }}</p>
        </div>
    @endslot

    @slot('footer', $product)
        <x-vaction-getx
            route="{{ route('products.show', $product->id) }}"
            icon="visibility"
            color="info"
        />
        <x-vaction-postx
            route="{{ route('cart.add', $product->id) }}"
            icon="add_shopping_cart"
            color="success"
        />
    @endslot
</x-vaction-card-list>
```

#### Quick Filter

```blade
<x-vaction-quick-filter
    :fields="[
        'name' => ['text', 'Name', 150],
        'email' => ['text', 'Email', 200],
        'status' => ['select', 'Status', [
            'active' => 'Active',
            'pending' => 'Pending',
            'blocked' => 'Blocked'
        ], 150]
    ]"
    route="{{ route('users.index') }}"
    :filters="{{ $request->all() }}"
/>
```

#### Quick Form

```blade
<x-vaction-quick-form
    title="Add New User"
    trigger-icon="fas fa-user-plus"
    trigger-color="success"
    trigger-label="Add User"
    size="medium"
    direction="right"
    overlay-color="dark"
>
    <form action="{{ route('users.store') }}" method="POST">
        @csrf
        <div class="field">
            <label class="label">Name</label>
            <div class="control">
                <input class="input" type="text" name="name" placeholder="User Name">
            </div>
        </div>

        <div class="field">
            <label class="label">Email</label>
            <div class="control">
                <input class="input" type="email" name="email" placeholder="user@example.com">
            </div>
        </div>

        <div class="field">
            <div class="control">
                <button type="submit" class="button is-primary">Save</button>
            </div>
        </div>
    </form>
</x-vaction-quick-form>
```

#### Collapsible Filter

The collapsible filter component works with Livewire to provide dynamic filtering. It requires a Livewire component to handle the filter state:

```blade
<x-vaction-collapsible-filter
    :filters="[
        'name' => ['text', 'Name'],
        'email' => ['text', 'Email Address'],
        'status' => ['select', 'Status', ['' => 'Select Status', 'active' => 'Active', 'inactive' => 'Inactive']],
        'date_range' => ['date_range', 'Registration Date']
    ]"
    :fields="[
        'name' => $filterValues['name'] ?? '',
        'email' => $filterValues['email'] ?? '',
        'status' => $filterValues['status'] ?? '',
        'date_range' => [
            'from' => $filterValues['date_range']['from'] ?? '',
            'to' => $filterValues['date_range']['to'] ?? ''
        ]
    ]"
    :field-hints="[
        'name' => 'Search by full or partial name',
        'email' => 'Search by full or partial email'
    ]"
    panel-label="Filter Users"
/>
```

#### Collapsible Filter Actions

```blade
<x-vaction-collapsible-filter-actions
    :filter-applied="$filtersApplied"
    label="Filter"
    icon="filter_list"
    panel="collapsible-filter-panel"
    reset-action="resetFilters"
/>
```

These components should be used within a Livewire component:

```php
class UsersList extends Component
{
    public $filterValues = [];
    public $filtersApplied = false;

    public function mount()
    {
        // Initialize filter values if needed
    }

    public function resetFilters()
    {
        $this->filterValues = [];
        $this->filtersApplied = false;
    }

    // Other methods to handle filtering
}
```

#### Button Components

The package includes several button components that simplify common UI actions:

##### Basic Button

```blade
<x-vaction-button
    icon="add"
    class="is-primary"
    size="normal"
>
    Create New
</x-vaction-button>
```

##### Icon Button

```blade
<x-vaction-button-icon
    icon="settings"
    color="info"
    action="openSettings"
    title="Settings"
    size="small"
/>
```

##### Edit Button

```blade
<x-vaction-button-edit
    action="edit({{ $item->id }})"
    title="Edit Item"
/>
```

##### Delete Button

```blade
<x-vaction-button-delete
    action="delete({{ $item->id }})"
    confirmMessage="Are you sure you want to delete this item?"
    title="Delete Item"
/>
```

##### Archive Variant of Delete Button

```blade
<x-vaction-button-delete
    action="archive({{ $item->id }})"
    isArchive="true"
    title="Archive Item"
/>
```

##### Show Button

```blade
<x-vaction-button-show
    action="showDetails({{ $item->id }})"
    title="View Details"
/>
```

##### Close Button

```blade
<x-vaction-button-close
    action="closeModal"
    title="Close"
    size="small"
/>
```

##### Scroll Up Button

The scroll-up button component provides a floating action button that allows users to smoothly scroll back to the top of the page or to a specific target element. It's positioned as a fixed overlay and can be customized for appearance and behavior:

```blade
<!-- Basic usage - scroll to page top -->
<x-vaction-button-scroll-up />
```

```blade
<!-- Scroll to specific target with custom styling -->
<x-vaction-button-scroll-up
    target="#header"
    :z-index="999"
    :size="48"
    color="primary"
    position="left"
    title="Back to top"
/>
```

```blade
<!-- With dynamic values from variables -->
<x-vaction-button-scroll-up
    :target="$scrollTarget"
    :z-index="$zIndex"
    :size="$buttonSize"
    :color="$theme"
    :position="$buttonPosition"
    :title="$tooltipText"
/>
```

**Available Properties:**

- `target` (string|null): CSS selector or element ID to scroll to. If null, scrolls to page top
- `z-index` (int): CSS z-index value for button layering (default: 4)
- `size` (int): Button size in pixels (width and height) (default: 64)
- `color` (string): Button color theme from Bulma colors (default: 'link')
- `position` (string): Screen position - 'left' or 'right' (default: 'right')
- `title` (string): Tooltip text displayed on hover (default: 'Scroll to top')

#### Save Cancel Component

The Actions Save component provides a standardized way to add save and cancel buttons to forms, especially useful in Livewire components:

```blade
<x-vaction-save-cancel
    save="Save Changes"
    saveAction="saveForm"
    saveIcon="save"
    cancel="Go Back"
    cancelAction="cancelForm"
    cancelIcon="arrow_back"
    size="are-medium"
/>
```

You can hide either the save or cancel button by setting their value to `false`:

```blade
<x-vaction-save-cancel
    cancel="false"
    saveAction="submitForm"
/>
```

You can also add additional buttons by using the slot:

```blade
<x-vaction-save-cancel
    saveAction="saveAndContinue"
    cancelAction="cancel"
>
    <button class="button is-info" type="button" wire:click="preview">
        <span class="icon">
            <span class="material-symbols-outlined">preview</span>
        </span>
        <span>Preview</span>
    </button>
</x-vaction-save-cancel>
```

## Icon Support

The components support both Material Symbols and Font Awesome icons. For Material Symbols, simply use the icon name. For Font Awesome, use the full class name starting with `fa`:

```blade
<!-- Material Symbols -->
<x-vaction-getx
    route="{{ route('dashboard') }}"
    icon="dashboard"
/>

<!-- Font Awesome -->
<x-vaction-getx
    route="{{ route('dashboard') }}"
    icon="fa fa-home"
/>
```

## Color Options

Components support all Bulma colors. You can use either the full class name (`is-primary`) or just the color name (`primary`):

- `primary`
- `info`
- `success`
- `warning`
- `danger`
- `link`
- `dark`
- `light`
- `white`
- `black`

## Browser Compatibility

This package is compatible with modern browsers that support ES6 JavaScript and CSS Flexbox:

- Chrome (latest)
- Firefox (latest)
- Safari (latest)
- Edge (latest)

## License

Proprietary - © 2025 V360 Tech. All rights reserved.

## Support

For questions, issues, or feature requests, please contact the development team at it@v360.tech.

## Changelog

- **17 Oct 2025**: Added documentation for SaveCancel component and new button-based components including Button, Button Icon, Button Edit, Button Delete, Button Show, and Button Close.
- **29 Sept 2025**: Updated documentation for Collapsible Filter and Quick Form components to accurately reflect their implementation.
- **Initial Release**: Basic documentation for all Action Components.
